/*
 * Copyright (C) 2016 Apple Inc. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY APPLE INC. AND ITS CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL APPLE INC. OR ITS CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF
 * THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include "MockGamepad.h"

#if ENABLE(GAMEPAD)

namespace WebCore {

MockGamepad::MockGamepad(unsigned index, const String& gamepadID, const String& mapping, unsigned axisCount, unsigned buttonCount, bool supportsDualRumble)
    : PlatformGamepad(index)
{
    m_connectTime = m_lastUpdateTime = MonotonicTime::now();
    updateDetails(gamepadID, mapping, axisCount, buttonCount, supportsDualRumble);
}

void MockGamepad::updateDetails(const String& gamepadID, const String& mapping, unsigned axisCount, unsigned buttonCount, bool supportsDualRumble)
{
    m_id = gamepadID;
    m_mapping = mapping;
    m_axisValues.clear();
    for (size_t i = 0; i < axisCount; ++i)
        m_axisValues.append({ });
    m_buttonValues.clear();
    for (size_t i = 0; i < buttonCount; ++i)
        m_buttonValues.append({ });
    m_lastUpdateTime = MonotonicTime::now();
    if (supportsDualRumble)
        m_supportedEffectTypes.add(GamepadHapticEffectType::DualRumble);
    else
        m_supportedEffectTypes.remove(GamepadHapticEffectType::DualRumble);
}

bool MockGamepad::setAxisValue(unsigned index, double value)
{
    if (index >= m_axisValues.size()) {
        LOG_ERROR("MockGamepad (%u): Attempt to set value on axis %u which doesn't exist", m_index, index);
        return false;
    }

    m_axisValues[index].setValue(value);
    m_lastUpdateTime = MonotonicTime::now();
    return true;
}

bool MockGamepad::setButtonValue(unsigned index, double value)
{
    if (index >= m_buttonValues.size()) {
        LOG_ERROR("MockGamepad (%u): Attempt to set value on button %u which doesn't exist", m_index, index);
        return false;
    }

    m_buttonValues[index].setValue(value);
    m_lastUpdateTime = MonotonicTime::now();
    return true;
}

} // namespace WebCore

#endif // ENABLE(GAMEPAD)
