/*
** (c) 1996-2000 The Regents of the University of California (through
** E.O. Lawrence Berkeley National Laboratory), subject to approval by
** the U.S. Department of Energy.  Your use of this software is under
** license -- the license agreement is attached and included in the
** directory as license.txt or you may contact Berkeley Lab's Technology
** Transfer Department at TTD@lbl.gov.  NOTICE OF U.S. GOVERNMENT RIGHTS.
** The Software was developed under funding from the U.S. Government
** which consequently retains certain rights as follows: the
** U.S. Government has been granted for itself and others acting on its
** behalf a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, and perform publicly
** and display publicly.  Beginning five (5) years after the date
** permission to assert copyright is obtained from the U.S. Department of
** Energy, and subject to any subsequent five (5) year renewals, the
** U.S. Government is granted for itself and others acting on its behalf
** a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, distribute copies to
** the public, perform publicly and display publicly, and to permit
** others to do so.
*/

#ifndef _MG_
#define _MG_

//
//@Man:
//@Memo: Multigrid class
/*@Doc:
  This multigrid class provides the framework for doing the different
  multigrid solves.  It provides virtual versions of the worker functions
  necessary for implementation of a solver: residual, 
  step (relax or solve), restrict, and interpolate.  
  It also allows for several types of multigrid cycles : 
  V-cycles, W-Cycles, or FMV-cycles (Full Multigrid V-cycles)
*/

class multigrid {
  // This friend declaration compensates for a compiler bug.
  friend class diffuser_mg;
  friend class macprojection_mg;
  friend class hgprojection_mg;

 protected:

  Box domain;
  FArrayBox * phi; 
  FArrayBox * source;
  FArrayBox * resid;
  Real _hx;
  Real _hy;
  Real _hz;
  multigrid *next;

//@ManDoc: Is this the first instantiation of multigrid?
  static int first_mg;

//@ManDoc: If true, print additional run-time diagnostics
  static int debug;

//@ManDoc: Used to set degrees of verbosity (can be 0,1 or 2)
  static int verbose;

//@ManDoc: Problem norm to be used to define convergence.
  static Real prob_norm;

//@ManDoc: Compute the residual R = RHS - DG(phi)
  virtual Real residual() = 0;

//@ManDoc: Relax on the residual equation, updating phi
  virtual void step(int) = 0;

//@ManDoc: Coarsen the residual to the next coarser multigrid level
  virtual void Restrict() = 0;

//@ManDoc: Interpolate the solution from the next coarser multigrid level
  virtual void interpolate() = 0;

//@ManDoc: Coordinates of the low side of the domain
  const int * lo_mg;

//@ManDoc: Coordinates of the high side of the domain
  const int * hi_mg;

 public:

//@ManDoc: Constructor
  multigrid(const Box & Domain, 
            FArrayBox * Phi, 
            FArrayBox * Source, 
            FArrayBox * Resid, 
            Real Hx,
            Real Hy,
            Real Hz);

//@ManDoc: Solve the linear equation by whichever means
  void  solve(Real tol, 
              Real prob_norm,
              int nngsrb, 
              int i2=2);

//@ManDoc: Perform a Full Multigrid V-Cycle (FMV)
  Real    fmv(Real tol, 
              int nngsrb, 
              int i2=2); 

//@ManDoc: Perform a single V-Cycle 
  Real vcycle(Real tol, 
              int nngsrb, 
              int i2=2);

//@ManDoc: Perform a single W-Cycle 
  Real wcycle(int nngsrb, 
              int i2=2);
};

#endif
