# This file is part of CycloneDX Python Library
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
# SPDX-License-Identifier: Apache-2.0
# Copyright (c) OWASP Foundation. All Rights Reserved.

"""CVSS related utilities"""

__all__ = ['vs_from_cvss_scores']

from typing import Union

from ...model.vulnerability import VulnerabilitySeverity


def vs_from_cvss_scores(scores: Union[tuple[float, ...], float, None]) -> VulnerabilitySeverity:
    """
    Derives the Severity of a Vulnerability from it's declared CVSS scores.

    Args:
        scores: A `tuple` of CVSS scores. CVSS scoring system allows for up to three separate scores.

    Returns:
        Always returns an instance of :class:`cyclonedx.model.vulnerability.VulnerabilitySeverity`.
    """
    if type(scores) is float:
        scores = (scores,)

    if scores is None:
        return VulnerabilitySeverity.UNKNOWN

    max_cvss_score: float
    if isinstance(scores, tuple):
        max_cvss_score = max(scores)
    else:
        max_cvss_score = float(scores)

    if max_cvss_score >= 9.0:
        return VulnerabilitySeverity.CRITICAL
    elif max_cvss_score >= 7.0:
        return VulnerabilitySeverity.HIGH
    elif max_cvss_score >= 4.0:
        return VulnerabilitySeverity.MEDIUM
    elif max_cvss_score > 0.0:
        return VulnerabilitySeverity.LOW
    else:
        return VulnerabilitySeverity.NONE
