/***********************************************************************************

    Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

    This file is part of Lifeograph.

    Lifeograph is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    Lifeograph is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include "panel_main.hpp"
#include "lifeograph.hpp"
#include "app_window.hpp"


using namespace LIFEO;


PanelMain::PanelMain( void )
:   m_bhist_index( 0 ), m_flag_index_locked( false ), m_flag_title_edit_in_progress( false ),
    m_ptr2widget( NULL ), m_entry_elem_title( NULL )
{
    Lifeograph::builder->get_widget( "vbox_panel_main", m_vbox_content );
    Lifeograph::builder->get_widget( "align_header_title", m_align_elem_title );
    Lifeograph::builder->get_widget( "eventbox_header_title", m_eventbox_elem_title );
    Lifeograph::builder->get_widget( "label_header_title", m_label_elem_title );
    Lifeograph::builder->get_widget( "label_header_subtitle", m_label_elem_extra );
    Lifeograph::builder->get_widget( "button_header_back", m_button_back );
    Lifeograph::builder->get_widget( "image_header_elem", m_image_elem );
    Menubutton::s_builder_name = "";
    Lifeograph::builder->get_widget_derived( "menubutton_header_elem", m_button_menu );
    Lifeograph::builder->get_widget( "alignment_header_extra_tools", m_alignment_extra_tools );

    m_entry_elem_title = new Gtk::Entry;

    for( int i = 0; i < BROWSING_HISTORY_SIZE; i++ )
        m_browsing_history[ i ] = NULL;

    m_entry_elem_title->set_icon_activatable( true, Gtk::ENTRY_ICON_SECONDARY );

    m_eventbox_elem_title->signal_button_press_event().connect(
            sigc::mem_fun( this, &PanelMain::handle_title_click ) );
    m_eventbox_elem_title->signal_enter_notify_event().connect(
            sigc::mem_fun( this, &PanelMain::handle_title_hover ) );
    m_eventbox_elem_title->signal_leave_notify_event().connect(
            sigc::mem_fun( this, &PanelMain::handle_title_hover_out ) );
    m_entry_elem_title->signal_changed().connect(
            sigc::mem_fun( this, &PanelMain::handle_entry_title_changed ) );
    m_entry_elem_title->signal_key_press_event().connect(
            sigc::mem_fun( this, &PanelMain::handle_entry_title_key_press ) );
    m_entry_elem_title->signal_focus_out_event().connect(
            sigc::mem_fun( this, &PanelMain::handle_entry_title_focus_out ) );
    m_entry_elem_title->signal_icon_release().connect(
            sigc::mem_fun( this, &PanelMain::handle_entry_icon_release ) );
    m_entry_elem_title->signal_activate().connect(
            sigc::mem_fun( this, &PanelMain::apply_title_edit ) );

    Lifeograph::create_action(
            false, m_action_previous_browsing, "BrowseBack", Gtk::Stock::GO_BACK,
            "", "",
            Gtk::AccelKey( GDK_KEY_Left, Gdk::MOD1_MASK ),
            sigc::mem_fun( this, &PanelMain::go_back ) );

    m_button_back->set_related_action( m_action_previous_browsing );
}

void
PanelMain::show( ElementShowerProto *shower )
{
    if( m_flag_title_edit_in_progress )
        m_button_back->grab_focus();    // cancel editing

    // prepare last element for hiding:
    if( m_browsing_history[ m_bhist_index ] )
        m_browsing_history[ m_bhist_index ]->prepare_for_hiding();

    // add new element to the history:
    if( ! m_flag_index_locked ) // locked while going back
    {
        // TODO: consider duplicate elements?
        m_bhist_index = ( m_bhist_index + 1 ) % BROWSING_HISTORY_SIZE;
        m_browsing_history[ m_bhist_index ] = shower->get_element();
    }

    m_cur_shower = shower;

    Lifeograph::m_internaloperation++;

    // HEADER TEXTS
    m_label_elem_title->set_text( shower->get_title_str() );
    m_label_elem_title->set_tooltip_markup( shower->get_tip_str() );
    m_label_elem_extra->set_text( shower->get_info_str() );

    // ICON
    m_image_elem->set( shower->get_element()->get_icon32() );

    // TOOLBAR
    m_button_menu->set_menu( shower->get_menu() );
    set_extra_tools( shower->get_extra_tools() );

    // CONTENT
    if( m_ptr2widget != shower->get_widget() )
    {
        if( m_ptr2widget )
        {
            m_vbox_content->remove( *m_ptr2widget );
        }
        m_vbox_content->pack_start( *shower->get_widget() );
        m_ptr2widget = shower->get_widget();
    }

    m_flag_title_editable = ( m_cur_shower->is_title_editable() && ! Diary::d->is_read_only() );

    AppWindow::p->panel_diary->show( shower->get_element() );

    Lifeograph::m_internaloperation--;
}

void
PanelMain::handle_logout( void )
{
    if( m_flag_title_edit_in_progress )
    {
        m_flag_title_edit_in_progress = false;  // must be first

        m_align_elem_title->remove();
        m_align_elem_title->add( *m_eventbox_elem_title );
    }

    clear_history();
}

void
PanelMain::set_extra_tools( Gtk::Widget *tool )
{
    if( m_alignment_extra_tools->get_child() )
        m_alignment_extra_tools->remove();

    if( tool != NULL )
        m_alignment_extra_tools->add( *tool );
}

void
PanelMain::go_back( void )
{
    for( unsigned int i = ( m_bhist_index - 1 + BROWSING_HISTORY_SIZE ) % BROWSING_HISTORY_SIZE;
         i != m_bhist_index;
         i = ( i - 1 + BROWSING_HISTORY_SIZE ) % BROWSING_HISTORY_SIZE )
    {
        if( m_browsing_history[ i ] != NULL )
        {
            m_flag_index_locked = true;
            m_browsing_history[ i ]->show();
            m_bhist_index = i;
            m_flag_index_locked = false;
            return;
        }
    }
}

void
PanelMain::remove_element_from_history( DiaryElement *element )
{
    // TODO: this function must be improved
    for( int i = 0; i < BROWSING_HISTORY_SIZE; i++ )
    {
        if( m_browsing_history[ i ] == element )
            m_browsing_history[ i ] = NULL;
    }
}

void
PanelMain::clear_history( void )
{
    for( int i = 0; i < BROWSING_HISTORY_SIZE; i++ )
        m_browsing_history[ i ] = NULL;
}

void
PanelMain::start_title_edit( void )
{
    if( m_flag_title_editable )
    {
        m_align_elem_title->remove();

        m_align_elem_title->add( *m_entry_elem_title );
        m_entry_elem_title->show();
        m_entry_elem_title->set_text( m_cur_shower->get_title_edit_str() );
        m_entry_elem_title->grab_focus();
        m_entry_elem_title->set_icon_from_stock( Gtk::Stock::DIALOG_WARNING,
                Gtk::ENTRY_ICON_SECONDARY );

        m_flag_title_applicable = false;
        m_flag_title_edit_in_progress = true;
    }
}

void
PanelMain::cancel_title_edit( void )
{
    if( m_flag_title_edit_in_progress )
    {
        m_flag_title_edit_in_progress = false;  // must be first

        m_align_elem_title->remove();
        m_align_elem_title->add( *m_eventbox_elem_title );
        m_label_elem_title->set_text( m_cur_shower->get_title_str() );
    }
}

void
PanelMain::handle_entry_title_changed( void )
{
    if( m_flag_title_edit_in_progress )
    {
        m_flag_title_applicable = m_cur_shower->check_title_applicable(
                m_entry_elem_title->get_text() );

        m_entry_elem_title->set_icon_from_stock( m_flag_title_applicable ?
                Gtk::Stock::APPLY : Gtk::Stock::DIALOG_WARNING,
                Gtk::ENTRY_ICON_SECONDARY );
    }
}

void
PanelMain::apply_title_edit( void )
{
    if( m_flag_title_applicable )
    {
        Glib::ustring str( m_entry_elem_title->get_text() );

        m_button_back->grab_focus();    // cancel editing

        m_cur_shower->apply_title( str );

        m_label_elem_title->set_text( m_cur_shower->get_title_str() );
    }
}

bool
PanelMain::handle_title_click( GdkEventButton* )
{
    if( m_flag_title_editable )
        start_title_edit();
    return false;
}

// cancels editing:
bool
PanelMain::handle_entry_title_focus_out( GdkEventFocus* )
{
    cancel_title_edit();

    return false;
}

bool
PanelMain::handle_entry_title_key_press( GdkEventKey *event )
{
    if( event->keyval == GDK_KEY_Escape )
        m_button_back->grab_focus();    // cancel editing
    return true;
}

bool
PanelMain::handle_title_hover( GdkEventCrossing* )
{
    if( m_flag_title_editable )
        m_label_elem_title->set_markup(
                "<u>" + Glib::Markup::escape_text( m_cur_shower->get_title_str() ) + "</u>" );
    return false;
}

bool
PanelMain::handle_title_hover_out( GdkEventCrossing* )
{
    if( m_flag_title_editable )
        m_label_elem_title->set_text( m_cur_shower->get_title_str() );
    return false;
}

void
PanelMain::handle_entry_icon_release( Gtk::EntryIconPosition icon_position,
                                      const GdkEventButton* event )
{
    if( m_flag_title_applicable )
        apply_title_edit();
    else
        cancel_title_edit();
}
