#include "driftcorr.h"
#include "data.h"
#include "controller.h"



bool RecoDriftCalc::process(RecoData& rd, RecoController& controller) {
  Log<Reco> odinlog(c_label(),"process");

  Range all=Range::all();

  ComplexData<6>& indata=rd.data(Rank<6>());
  ComplexData<1>& outdata=rd.data(Rank<1>());

  TinyVector<int,6> inshape=indata.shape();
  ODINLOG(odinlog,normalDebug) << "inshape=" << inshape << STD_endl;

  int nchan=inshape(0);
  int nrep=inshape(1);
  int nslice=inshape(2);

  ComplexData<1> cplxtcourse(nrep);
  Data<float,1> phasecourse(nrep);
  Data<float,1> phasecoursesum(nrep); phasecoursesum=0.0;
  float magnsum=0.0;
  for(int ichan=0; ichan<nchan; ichan++) {
    for(int islice=0; islice<nslice; islice++) {
      for(int irep=0; irep<nrep; irep++) cplxtcourse(irep)=sum(indata(ichan,irep,islice,all,all,all));

      phasecourse=unwrap_phase(phase(cplxtcourse),0);
      float magn=sum(cabs(cplxtcourse));
      phasecoursesum+=magn*phasecourse;
      magnsum+=magn;

    }
  }

  phasecoursesum-=phasecoursesum(0);

  float te=controller.protocol().seqpars.get_EchoTime();

  if(magnsum>0.0 && te>0.0) {
    rd.mode=RecoData::real_data;
    outdata.resize(nrep);
    outdata=float2real(phasecoursesum/(magnsum*te));
  } else {
    ODINLOG(odinlog,errorLog) << "Cannot calculate field drift" << STD_endl;
    return false;
  }

  indata.free();

  return execute_next_step(rd,controller);
}



/////////////////////////////////////////////////////////////////////////



bool RecoDriftCorr::process(RecoData& rd, RecoController& controller) {
  Log<Reco> odinlog(c_label(),"process");

  double freq=0.0;
  mutex.lock();
  int cachesize=driftcache.size();
  mutex.unlock();

  ODINLOG(odinlog,normalDebug) << "cachesize=" << cachesize << STD_endl;

  if(!cachesize) {

    RecoData rdfdrift;
    if(!controller.inquire_data(posted_drift_str, rdfdrift)) return false;

    mutex.lock();
    driftcache.reference(creal(rdfdrift.data(Rank<1>())));
    ODINLOG(odinlog,normalDebug) << "driftcache" << driftcache << STD_endl;
    mutex.unlock();
  }

  int irep=rd.coord().index[repetition];
  mutex.lock();
  int nrep=driftcache.size();
  if(irep<nrep) {
    freq=driftcache(irep);
  } else {
    ODINLOG(odinlog,errorLog) << "repetition index = " << irep << " out of range (0," << nrep << ")" << STD_endl;
    return false;
  }
  mutex.unlock();


  if(freq) {

    ComplexData<1>& data=rd.data(Rank<1>());
    unsigned int adcSize=data.extent(0);
    double dt=rd.coord().dwellTime;
    ODINLOG(odinlog,normalDebug) << "freq/adcSize/dt=" << freq << "/" << adcSize << "/" << dt << STD_endl;

    double reftime=0.0;
    dvector echostart=controller.dim_values(echo);
    unsigned int iecho=rd.coord().echopos;
    if(echostart.size()) reftime=echostart[iecho];
    ODINLOG(odinlog,normalDebug) << "echostart/reftime=" << echostart << "/" << reftime << STD_endl;

    for(unsigned int isample=0; isample<adcSize; isample++) {
      double time=reftime+double(isample)*dt;
      double imag=-time*freq;
      STD_complex phasefactor=expc(float2imag(imag));
      data(isample)*=phasefactor;
    }

  }

  return execute_next_step(rd,controller);
}
