// @HEADER
// *****************************************************************************
//        MueLu: A package for multigrid based preconditioning
//
// Copyright 2012 NTESS and the MueLu contributors.
// SPDX-License-Identifier: BSD-3-Clause
// *****************************************************************************
// @HEADER

#ifndef MUELU_MATRIXANALYSISFACTORY_DECL_HPP_
#define MUELU_MATRIXANALYSISFACTORY_DECL_HPP_

#include <Xpetra_Map_fwd.hpp>
#include <Xpetra_StridedMap_fwd.hpp>
#include <Xpetra_Vector_fwd.hpp>
#include <Xpetra_VectorFactory_fwd.hpp>
#include <Xpetra_Matrix_fwd.hpp>

#include "MueLu_ConfigDefs.hpp"
#include "MueLu_TwoLevelFactoryBase.hpp"
#include "MueLu_Utilities_fwd.hpp"

namespace MueLu {

/*!
  @class MatrixAnalysisFactory
  @brief This class checks matrix properties of A on current level.
  This factory can be plugged in everywhere
  in the factory layout of the MueLu preconditioner. It only needs A as input and pipes it through as output.

  ## Input/output of MatrixAnalysisFactory ##

  ### User parameters of MatrixAnalysisFactory ###
  Parameter | type | default | master.xml | validated | requested | description
  ----------|------|---------|:----------:|:---------:|:---------:|------------
  | A       | Factory | null  |   | * | * | Generating factory of the matrix A|

  The * in the @c master.xml column denotes that the parameter is defined in the @c master.xml file.<br>
  The * in the @c validated column means that the parameter is declared in the list of valid input parameters (see MatrixAnalysisFactory::GetValidParameters).<br>
  The * in the @c requested column states that the data is requested as input with all dependencies (see MatrixAnalysisFactory::DeclareInput).

  ### Variables provided by MatrixAnalysisFactory ###

  After MatrixAnalysisFactory::Build the following data is available (if requested)

  Parameter | generated by | description
  ----------|--------------|------------
  | A       | MatrixAnalysisFactory   | Original operator (without changes)


*/

template <class Scalar        = DefaultScalar,
          class LocalOrdinal  = DefaultLocalOrdinal,
          class GlobalOrdinal = DefaultGlobalOrdinal,
          class Node          = DefaultNode>
class MatrixAnalysisFactory : public TwoLevelFactoryBase {
#undef MUELU_MATRIXANALYSISFACTORY_SHORT
#include "MueLu_UseShortNames.hpp"

 public:
  //! @name Constructors/Destructors.
  //@{

  //! Constructor.
  MatrixAnalysisFactory();

  //! Destructor.
  virtual ~MatrixAnalysisFactory();

  //@}

  RCP<const ParameterList> GetValidParameterList() const;

  //! @name Input
  //@{

  /*! @brief Specifies the data that this class needs, and the factories that generate that data.

      If the Build method of this class requires some data, but the generating factory is not specified in DeclareInput, then this class
      will fall back to the settings in FactoryManager.
  */
  void DeclareInput(Level &fineLevel, Level &coarseLevel) const;

  //@}

  //! @name Build methods.
  //@{

  //! Build an object with this factory.
  void Build(Level &fineLevel, Level &coarseLevel) const;

  //@}

 private:
  void checkVectorEntries(const Teuchos::RCP<Vector> &vec, std::string info) const;

};  // class MatrixAnalysisFactory

}  // namespace MueLu

#define MUELU_MATRIXANALYSISFACTORY_SHORT

#endif /* MUELU_MATRIXANALYSISFACTORY_DECL_HPP_ */
