#! /usr/bin/env python3
# -*- coding: utf-8 -*-

#	Copyright 2013, Marten de Vries
#
#	This file is part of Code of Conduct Signing Assistant.
#
#	Code of Conduct Signing Assistant is free software: you can
#	redistribute it and/or modify it under the terms of the GNU General
#	Public License as published by the Free Software Foundation, either
#	version 3 of the License, or (at your option) any later version.
#
#	Code of Conduct Signing Assistant is distributed in the hope that it
#	will be useful, but WITHOUT ANY WARRANTY; without even the implied
#	warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#	See the GNU General Public License for more details.
#
#	You should have received a copy of the GNU General Public License
#	along with Code of Conduct Signing Assistant. If not, see
#	<http://www.gnu.org/licenses/>.

import urllib.request
import tempfile
import os
import subprocess
import sys
import re
import gettext

from PyQt4 import QtCore, QtGui, QtWebKit, QtNetwork
import gnupg

SUBSITE = ""

EDIT_PGP_URL = "https://%slaunchpad.net/people/+me/+editpgpkeys" % SUBSITE
COC_URL = "https://%slaunchpad.net/codeofconduct/2.0/+download" % SUBSITE
SIGN_URL = "https://%slaunchpad.net/codeofconduct/2.0/+sign" % SUBSITE
KEY_LENGTH = 2048

class IntroPage(QtGui.QWizardPage):
	def __init__(self, *args, **kwargs):
		super().__init__(*args, **kwargs)

		self._label = QtGui.QLabel()
		self._label.setWordWrap(True)

		vbox = QtGui.QVBoxLayout()
		vbox.addWidget(self._label)
		self.setLayout(vbox)

	def retranslate(self):
		self.setTitle(_("Introduction"))
		self._label.setText(_("This wizard will guide you through the process of signing the Ubuntu Code of Conduct. Click Next to start."))

class KeyModel(QtCore.QAbstractListModel):
	def update(self, gpg):
		self.beginResetModel()
		self._lists = gpg.list_keys()
		self.endResetModel()

	def rowCount(self, parent):
		return len(self._lists)

	def data(self, index, role):
		if not index.isValid():
			return
		item = self._lists[index.row()]
		if role == QtCore.Qt.DisplayRole:
			return ", ".join(item["uids"])
		elif role == QtCore.Qt.UserRole:
			return item["keyid"], item["fingerprint"]

class KeyView(QtGui.QListView):
	def __init__(self, *args, **kwargs):
		super().__init__(*args, **kwargs)

		self.setModel(KeyModel())

	def update(self, gpg):
		self.model().update(gpg)

class KeyChoicePage(QtGui.QWizardPage):
	def __init__(self, *args, **kwargs):
		super().__init__(*args, **kwargs)

		self._label = QtGui.QLabel()
		self._label.setWordWrap(True)
		self._view = KeyView()
		self._view.selectionModel().selectionChanged.connect(self._selectionChanged)
		self._button = QtGui.QPushButton()
		self._button.clicked.connect(self._newKey)

		vbox = QtGui.QVBoxLayout()
		vbox.addWidget(self._label)
		vbox.addWidget(self._view)
		vbox.addWidget(self._button)
		self.setLayout(vbox)

	def _reset(self):
		self.keyId = None
		self.fingerprint = None

	def _selectionChanged(self, newSelection, oldSelection):
		index = newSelection.indexes()[0]
		if index.isValid():
			data = index.data(QtCore.Qt.UserRole)
			self.keyId, self.fingerprint = data
		else:
			self._reset()
		self.completeChanged.emit()

	def _newKey(self):
		question = _("What's your name?")
		name, ok = QtGui.QInputDialog.getText(self, question, question)
		if not ok:
			return

		question2 = _("What's your email address?")
		email, ok2 = QtGui.QInputDialog.getText(self, question2, question2)
		if not ok2:
			return

		input_data = self.wizard().gpg.gen_key_input(
			key_type="RSA",
			key_length=KEY_LENGTH,
			name_real=name,
			name_comment=_("Generated by cocSigner.py"),
			name_email=email
		)
		self.wizard().gpg.gen_key(input_data)
		self.initializePage()

	def initializePage(self):
		self._view.update(self.wizard().gpg)
		self._reset()

	def retranslate(self):
		self.setTitle(_("Please choose a key"))
		self._label.setText(_("To sign the Code of Conduct, you need to have a PGP (Pretty Good Privacy) key. Choose an existing one below, or create one if you do not have one."))
		self._button.setText(_("Create new key"))

	def isComplete(self):
		return self.keyId is not None and self.fingerprint is not None

class PgpLaunchpadPage(QtGui.QWizardPage):
	def __init__(self,*args, **kwargs):
		super().__init__(*args, **kwargs)

		self._label = QtGui.QLabel()
		self._label.setWordWrap(True)

		self._fingerPrintLabelLabel = QtGui.QLabel()
		self._fingerPrintLabel = QtGui.QLabel()
		self._fingerPrintLabel.setTextInteractionFlags(QtCore.Qt.TextSelectableByMouse)

		self._webView = QtWebKit.QWebView()

		vbox = QtGui.QVBoxLayout()
		vbox.addWidget(self._label)
		vbox.addWidget(self._fingerPrintLabelLabel)
		vbox.addWidget(self._fingerPrintLabel)
		vbox.addWidget(self._webView)

		self.setLayout(vbox)

	def initializePage(self):
		fp = self.wizard().fingerprint
		self._fingerPrintLabel.setText(fp)
		self._fingerPrintLabel.setSelection(0, len(fp))
		QtGui.QApplication.instance().clipboard().setText(fp)

		self._webView.page().networkAccessManager().setCookieJar(self.wizard().cookieJar)
		self._webView.setUrl(QtCore.QUrl(EDIT_PGP_URL))

		subprocess.check_call(["gpg", "--send-keys", "--keyserver", "keyserver.ubuntu.com", self.wizard().keyId])

	def retranslate(self):
		self.setTitle(_("Add PGP key to Launchpad"))
		#TRANSLATORS: Don't translate 'Import key', because that part is
		#TRANSLATORS: a Launchpad button & Launchpad isn't available in
		#TRANSLATORS: any language beside English. Thanks.
		self._label.setText(_("Your PGP key needs to be imported into Launchpad before you can use it to sign the Code of Conduct. Please do so by copying your fingerprint (which is shown below) into the respective box in the website shown below, and click 'Import key'. When that's done, click 'Next'. Please note that it can take up to 10 minutes before Launchpad accepts your key. If so, just try again in a few minutes."))
		self._fingerPrintLabelLabel.setText(_("Fingerprint:"))

class DecryptMailPage(QtGui.QWizardPage):
	_re = re.compile(r"(https://" + SUBSITE + "launchpad.net/\S+)")

	def __init__(self, *args, **kwargs):
		super().__init__(*args, **kwargs)

		self._label = QtGui.QLabel()
		self._label.setWordWrap(True)
		self._textEdit = QtGui.QTextEdit()
		self._textEdit.textChanged.connect(self._decrypt)

		vbox = QtGui.QVBoxLayout()
		vbox.addWidget(self._label)
		vbox.addWidget(self._textEdit)

		self.setLayout(vbox)

	def initializePage(self):
		self.decryptedLink = None

	def retranslate(self):
		self.setTitle(_("Decrypt mail sent from Launchpad"))
		self._label.setText(_("Launchpad has sent you an encrypted mail to verify you really uploaded the key. Please paste the part of the message that starts with '-----BEGIN PGP MESSAGE-----' and ends with '-----END PGP MESSAGE-----' in the text box below and click Next."))

	def _decrypt(self):
		encryptedText = str(self._textEdit.toPlainText())
		decrypted = self.wizard().gpg.decrypt(encryptedText)
		if decrypted.ok:
			#UTF-8 is a guess. It's ascii, but if it ever changes UTF-8
			#is the most likely candidate.
			decryptedText = str(decrypted.data, encoding="UTF-8")
			self.decryptedLink = self._re.search(decryptedText).group(0)
		else:
			self.decryptedLink = None
		self.completeChanged.emit()

	def isComplete(self):
		return self.decryptedLink is not None

class ConfirmLaunchpadPgpPage(QtGui.QWizardPage):
	def __init__(self,*args, **kwargs):
		super().__init__(*args, **kwargs)

		self._label = QtGui.QLabel()
		self._label.setWordWrap(True)

		self._webView = QtWebKit.QWebView()

		vbox = QtGui.QVBoxLayout()
		vbox.addWidget(self._label)
		vbox.addWidget(self._webView)

		self.setLayout(vbox)

	def initializePage(self):
		self._webView.page().networkAccessManager().setCookieJar(self.wizard().cookieJar)
		self._webView.setUrl(QtCore.QUrl(self.wizard().lpConfirmationLink))

	def retranslate(self):
		self.setTitle(_("Confirm Launchpad PGP key upload"))
		#TRANSLATORS: Please leave 'Continue' in English
		self._label.setText(_("Please click 'Continue' so your PGP key is definitively added to your Launchpad account. When you've clicked 'Continue', click Next. Note that you don't need to download the Ubuntu Code of Conduct yourself, as Launchpad suggests."))

class SignCodeOfConductPage(QtGui.QWizardPage):
	def __init__(self,*args, **kwargs):
		super().__init__(*args, **kwargs)

		self._label = QtGui.QLabel()
		self._label.setWordWrap(True)

		self._textEdit = QtGui.QTextEdit()

		vbox = QtGui.QVBoxLayout()
		vbox.addWidget(self._label)
		vbox.addWidget(self._textEdit)

		self.setLayout(vbox)

	def initializePage(self):
		response = urllib.request.urlopen(COC_URL)
		#same educated guess.
		self._codeOfConduct = str(response.read(), encoding="UTF-8")
		self._textEdit.setText(self._codeOfConduct)

	def retranslate(self):
		self.setTitle(_("Signing the Code of Conduct"))
		self._label.setText(_("Please take some time to read the Code of Conduct below. Are you sure you want to sign it? If so, click Next to do that."))

	@property
	def signedCodeOfConduct(self):
		sign = self.wizard().gpg.sign
		return sign(self._codeOfConduct, keyid=self.wizard().keyId).data

class SignaturePublishingPage(QtGui.QWizardPage):
	def __init__(self,*args, **kwargs):
		super().__init__(*args, **kwargs)

		self._label = QtGui.QLabel()
		self._label.setWordWrap(True)
		self._textEdit = QtGui.QTextEdit()

		self._webView = QtWebKit.QWebView()

		vbox = QtGui.QVBoxLayout()
		vbox.addWidget(self._label)
		vbox.addWidget(self._textEdit)
		vbox.addWidget(self._webView)

		self.setLayout(vbox)

	def initializePage(self):
		#same guess.
		signedCoc = str(self.wizard().signedCodeOfConduct, encoding="UTF-8")
		self._textEdit.setText(signedCoc)
		self._textEdit.selectAll()
		QtGui.QApplication.instance().clipboard().setText(signedCoc)

		self._webView.page().networkAccessManager().setCookieJar(self.wizard().cookieJar)
		self._webView.setUrl(QtCore.QUrl(SIGN_URL))

	def retranslate(self):
		self.setTitle(_("Publish Code of Conduct signature"))
		self._label.setText(_("Now publish your Code of Conduct signature by copying it from the text box above into the destined box in the website below, and clicking the 'Continue' button on that website. Then, click Next."))

class CongratulationsPage(QtGui.QWizardPage):
	def __init__(self,*args, **kwargs):
		super().__init__(*args, **kwargs)

		self._label = QtGui.QLabel()
		self._label.setWordWrap(True)

		vbox = QtGui.QVBoxLayout()
		vbox.addWidget(self._label)
		self.setLayout(vbox)

	def retranslate(self):
		self.setTitle(_("Congratulations!"))
		self._label.setText(_("Congratulations! You succesfully signed the code of conduct. You're an Ubuntero now! Please click Finish to close this program."))

class SideWidget(QtGui.QWidget):
	def __init__(self, wizard, iconPath, *args, **kwargs):
		super().__init__(*args, **kwargs)

		self._wizard = wizard

		pixmapLabel = QtGui.QLabel()
		pixmap = QtGui.QPixmap(iconPath).scaledToWidth(128, QtCore.Qt.SmoothTransformation)
		pixmapLabel.setPixmap(pixmap)
		self._textLabel = QtGui.QLabel()

		layout = QtGui.QVBoxLayout()
		layout.addWidget(pixmapLabel)
		layout.addStretch()
		layout.addWidget(self._textLabel)
		layout.addStretch()
		self.setLayout(layout)

	def retranslate(self):
		#TRANSLATORS: Please leave {current} and {total} in English,
		#TRANSLATORS: the program replaces them with numbers.
		self._textLabel.setText(_("Page {current}/{total}").format(
			current=self._wizard.currentPageNumber,
			total=self._wizard.totalAmountOfPages,
		))

	update = retranslate

class Wizard(QtGui.QWizard):
	def __init__(self, *args, **kwargs):
		super().__init__(*args, **kwargs)

		self._keyChoicePage = KeyChoicePage()
		self._decryptMailPage = DecryptMailPage()
		self._signCodeOfConductPage = SignCodeOfConductPage()
		self._pages = [
			IntroPage(),
			self._keyChoicePage,
			PgpLaunchpadPage(),
			self._decryptMailPage,
			ConfirmLaunchpadPgpPage(),
			self._signCodeOfConductPage,
			SignaturePublishingPage(),
			CongratulationsPage(),
		]

		self.cookieJar = QtNetwork.QNetworkCookieJar()

		for page in self._pages:
			self.addPage(page)

		self.gpg = gnupg.GPG()

		icon = "codeOfConductSigningAssistant.svg"
		self.setWindowIcon(QtGui.QIcon(icon))

		self._sideWidget = SideWidget(self, icon)
		self.currentIdChanged.connect(self._sideWidget.update)
		self.setSideWidget(self._sideWidget)

	@property
	def currentPageNumber(self):
		try:
			return self._pages.index(self.currentPage()) + 1
		except ValueError:
			return 0

	@property
	def totalAmountOfPages(self):
		return len(self._pages)

	@property
	def keyId(self):
		return self._keyChoicePage.keyId

	@property
	def fingerprint(self):
		return self._keyChoicePage.fingerprint

	@property
	def lpConfirmationLink(self):
		return self._decryptMailPage.decryptedLink

	@property
	def signedCodeOfConduct(self):
		return self._signCodeOfConductPage.signedCodeOfConduct

	def retranslate(self):
		self.setWindowTitle(_("Code of Conduct Signing Assistant"))
		self._sideWidget.retranslate()
		for page in self._pages:
			page.retranslate()

def main():
	try:
		#Try to load translations from the dev location
		t = gettext.translation("code-of-conduct-signing-assistant", localedir="../translations")
		t.install(names=['ngettext'])
	except IOError:
		#If that failed, try the system location
		t = gettext.translation("code-of-conduct-signing-assistant", fallback=True)
		t.install(names=["ngettext"])

	app = QtGui.QApplication(sys.argv)
	wizard = Wizard()
	wizard.retranslate()
	wizard.showMaximized()
	app.exec_()

if __name__ == "__main__":
	main()
