/* -*- Mode: C; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim:set sw=2 sts=2 et cin: */
/* 
 * This file is part of the MUSE Instrument Pipeline
 * Copyright (C) 2005-2015 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

/* This file was automatically generated */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*
 *                              Includes                                      *
 *----------------------------------------------------------------------------*/
#include <string.h> /* strcmp(), strstr() */
#include <strings.h> /* strcasecmp() */
#include <cpl.h>

#include "muse_scipost_subtract_sky_z.h" /* in turn includes muse.h */

/*----------------------------------------------------------------------------*/
/**
  @defgroup recipe_muse_scipost_subtract_sky         Recipe muse_scipost_subtract_sky: Subtract night sky model.
  @author Ole Streicher
  Subtract the sky as defined by the sky lines and continuum from a pixel table. This is a separated task of muse_scipost.
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*
 *                         Static variables                                   *
 *----------------------------------------------------------------------------*/
static const char *muse_scipost_subtract_sky_help =
  "Subtract the sky as defined by the sky lines and continuum from a pixel table. This is a separated task of muse_scipost.";

static const char *muse_scipost_subtract_sky_help_esorex =
  "\n\nInput frames for raw frame tag \"PIXTABLE_REDUCED\":\n"
  "\n Frame tag            Type Req #Fr Description"
  "\n -------------------- ---- --- --- ------------"
  "\n PIXTABLE_REDUCED     raw   Y      Flux calibrated input pixel table(s)."
  "\n SKY_LINES            calib Y    1 Sky line list"
  "\n SKY_CONTINUUM        calib Y    1 Sky continuum spectrum"
  "\n LSF_PROFILE          calib Y      LSF for each IFU."
  "\n\nProduct frames for raw frame tag \"PIXTABLE_REDUCED\":\n"
  "\n Frame tag            Level    Description"
  "\n -------------------- -------- ------------"
  "\n PIXTABLE_REDUCED     final    Output pixel table(s) for sky subtraction.";

/*----------------------------------------------------------------------------*/
/**
  @private
  @brief   Create the recipe config for this plugin.

  @remark The recipe config is used to check for the tags as well as to create
          the documentation of this plugin.
 */
/*----------------------------------------------------------------------------*/
static cpl_recipeconfig *
muse_scipost_subtract_sky_new_recipeconfig(void)
{
  cpl_recipeconfig *recipeconfig = cpl_recipeconfig_new();
      
  cpl_recipeconfig_set_tag(recipeconfig, "PIXTABLE_REDUCED", 1, -1);
  cpl_recipeconfig_set_input(recipeconfig, "PIXTABLE_REDUCED", "SKY_LINES", 1, 1);
  cpl_recipeconfig_set_input(recipeconfig, "PIXTABLE_REDUCED", "SKY_CONTINUUM", 1, 1);
  cpl_recipeconfig_set_input(recipeconfig, "PIXTABLE_REDUCED", "LSF_PROFILE", 1, -1);
  cpl_recipeconfig_set_output(recipeconfig, "PIXTABLE_REDUCED", "PIXTABLE_REDUCED");
    
  return recipeconfig;
} /* muse_scipost_subtract_sky_new_recipeconfig() */

/*----------------------------------------------------------------------------*/
/**
  @private
  @brief   Return a new header that shall be filled on output.
  @param   aFrametag   tag of the output frame
  @param   aHeader     the prepared FITS header
  @return  CPL_ERROR_NONE on success another cpl_error_code on error

  @remark This function is also used to generate the recipe documentation.
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code
muse_scipost_subtract_sky_prepare_header(const char *aFrametag, cpl_propertylist *aHeader)
{
  cpl_ensure_code(aFrametag, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(aHeader, CPL_ERROR_NULL_INPUT);
  if (!strcmp(aFrametag, "PIXTABLE_REDUCED")) {
  } else {
    cpl_msg_warning(__func__, "Frame tag %s is not defined", aFrametag);
    return CPL_ERROR_ILLEGAL_INPUT;
  }
  return CPL_ERROR_NONE;
} /* muse_scipost_subtract_sky_prepare_header() */

/*----------------------------------------------------------------------------*/
/**
  @private
  @brief   Return the level of an output frame.
  @param   aFrametag   tag of the output frame
  @return  the cpl_frame_level or CPL_FRAME_LEVEL_NONE on error

  @remark This function is also used to generate the recipe documentation.
 */
/*----------------------------------------------------------------------------*/
static cpl_frame_level
muse_scipost_subtract_sky_get_frame_level(const char *aFrametag)
{
  if (!aFrametag) {
    return CPL_FRAME_LEVEL_NONE;
  }
  if (!strcmp(aFrametag, "PIXTABLE_REDUCED")) {
    return CPL_FRAME_LEVEL_FINAL;
  }
  return CPL_FRAME_LEVEL_NONE;
} /* muse_scipost_subtract_sky_get_frame_level() */

/*----------------------------------------------------------------------------*/
/**
  @private
  @brief   Return the mode of an output frame.
  @param   aFrametag   tag of the output frame
  @return  the muse_frame_mode

  @remark This function is also used to generate the recipe documentation.
 */
/*----------------------------------------------------------------------------*/
static muse_frame_mode
muse_scipost_subtract_sky_get_frame_mode(const char *aFrametag)
{
  if (!aFrametag) {
    return MUSE_FRAME_MODE_ALL;
  }
  if (!strcmp(aFrametag, "PIXTABLE_REDUCED")) {
    return MUSE_FRAME_MODE_ALL;
  }
  return MUSE_FRAME_MODE_ALL;
} /* muse_scipost_subtract_sky_get_frame_mode() */

/*----------------------------------------------------------------------------*/
/**
  @private
  @brief   Setup the recipe options.
  @param   aPlugin   the plugin
  @return  0 if everything is ok, -1 if not called as part of a recipe.

  Define the command-line, configuration, and environment parameters for the
  recipe.
 */
/*----------------------------------------------------------------------------*/
static int
muse_scipost_subtract_sky_create(cpl_plugin *aPlugin)
{
  /* Check that the plugin is part of a valid recipe */
  cpl_recipe *recipe;
  if (cpl_plugin_get_type(aPlugin) == CPL_PLUGIN_TYPE_RECIPE) {
    recipe = (cpl_recipe *)aPlugin;
  } else {
    return -1;
  }

  /* register the extended processing information (new FITS header creation, *
   * getting of the frame level for a certain tag)                           */
  muse_processinginfo_register(recipe,
                               muse_scipost_subtract_sky_new_recipeconfig(),
                               muse_scipost_subtract_sky_prepare_header,
                               muse_scipost_subtract_sky_get_frame_level,
                               muse_scipost_subtract_sky_get_frame_mode);

  /* XXX initialize timing in messages                                       *
   *     since at least esorex is too stupid to turn it on, we have to do it */
  if (muse_cplframework() == MUSE_CPLFRAMEWORK_ESOREX) {
    cpl_msg_set_time_on();
  }

  /* Create the parameter list in the cpl_recipe object */
  recipe->parameters = cpl_parameterlist_new();
  /* Fill the parameters list */
  cpl_parameter *p;
      
  /* --lambdamin: Cut off the data below this wavelength after loading the pixel table(s). */
  p = cpl_parameter_new_value("muse.muse_scipost_subtract_sky.lambdamin",
                              CPL_TYPE_DOUBLE,
                             "Cut off the data below this wavelength after loading the pixel table(s).",
                              "muse.muse_scipost_subtract_sky",
                              (double)4000.);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CFG, "lambdamin");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "lambdamin");

  cpl_parameterlist_append(recipe->parameters, p);
      
  /* --lambdamax: Cut off the data above this wavelength after loading the pixel table(s). */
  p = cpl_parameter_new_value("muse.muse_scipost_subtract_sky.lambdamax",
                              CPL_TYPE_DOUBLE,
                             "Cut off the data above this wavelength after loading the pixel table(s).",
                              "muse.muse_scipost_subtract_sky",
                              (double)10000.);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CFG, "lambdamax");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "lambdamax");

  cpl_parameterlist_append(recipe->parameters, p);
      
  /* --orig: If specified, write an additional column containing the
          original data to the pixel table. */
  p = cpl_parameter_new_value("muse.muse_scipost_subtract_sky.orig",
                              CPL_TYPE_STRING,
                             "If specified, write an additional column containing the original data to the pixel table.",
                              "muse.muse_scipost_subtract_sky",
                              (const char *)"");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CFG, "orig");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "orig");

  cpl_parameterlist_append(recipe->parameters, p);
      
  /* --flux_sky: Reference flat field flux, obtained by sky exposure. This parameter is needed to scale the data of each pixel table if more than one pixel table was used to determine the sky. By default, it is taken from the parameter ESO DRS MUSE FLAT FLUX SKY of the first pixel table. */
  p = cpl_parameter_new_value("muse.muse_scipost_subtract_sky.flux_sky",
                              CPL_TYPE_DOUBLE,
                             "Reference flat field flux, obtained by sky exposure. This parameter is needed to scale the data of each pixel table if more than one pixel table was used to determine the sky. By default, it is taken from the parameter ESO DRS MUSE FLAT FLUX SKY of the first pixel table.",
                              "muse.muse_scipost_subtract_sky",
                              (double)0.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CFG, "flux_sky");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "flux_sky");

  cpl_parameterlist_append(recipe->parameters, p);
      
  /* --flux_lamp: Reference flat field flux, obtained by lamp exposure. This parameter is needed to scale the data of each pixel table if more than one pixel table was used to determine the sky. By default, it is taken from the parameter ESO DRS MUSE FLAT FLUX LAMP of the first pixel table. */
  p = cpl_parameter_new_value("muse.muse_scipost_subtract_sky.flux_lamp",
                              CPL_TYPE_DOUBLE,
                             "Reference flat field flux, obtained by lamp exposure. This parameter is needed to scale the data of each pixel table if more than one pixel table was used to determine the sky. By default, it is taken from the parameter ESO DRS MUSE FLAT FLUX LAMP of the first pixel table.",
                              "muse.muse_scipost_subtract_sky",
                              (double)0.0);
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CFG, "flux_lamp");
  cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "flux_lamp");

  cpl_parameterlist_append(recipe->parameters, p);
    
  return 0;
} /* muse_scipost_subtract_sky_create() */

/*----------------------------------------------------------------------------*/
/**
  @private
  @brief   Fill the recipe parameters into the parameter structure
  @param   aParams       the recipe-internal structure to fill
  @param   aParameters   the cpl_parameterlist with the parameters
  @return  0 if everything is ok, -1 if something went wrong.

  This is a convienience function that centrally fills all parameters into the
  according fields of the recipe internal structure.
 */
/*----------------------------------------------------------------------------*/
static int
muse_scipost_subtract_sky_params_fill(muse_scipost_subtract_sky_params_t *aParams, cpl_parameterlist *aParameters)
{
  cpl_ensure_code(aParams, CPL_ERROR_NULL_INPUT);
  cpl_ensure_code(aParameters, CPL_ERROR_NULL_INPUT);
  cpl_parameter *p;
      
  p = cpl_parameterlist_find(aParameters, "muse.muse_scipost_subtract_sky.lambdamin");
  cpl_ensure_code(p, CPL_ERROR_DATA_NOT_FOUND);
  aParams->lambdamin = cpl_parameter_get_double(p);
      
  p = cpl_parameterlist_find(aParameters, "muse.muse_scipost_subtract_sky.lambdamax");
  cpl_ensure_code(p, CPL_ERROR_DATA_NOT_FOUND);
  aParams->lambdamax = cpl_parameter_get_double(p);
      
  p = cpl_parameterlist_find(aParameters, "muse.muse_scipost_subtract_sky.orig");
  cpl_ensure_code(p, CPL_ERROR_DATA_NOT_FOUND);
  aParams->orig = cpl_parameter_get_string(p);
      
  p = cpl_parameterlist_find(aParameters, "muse.muse_scipost_subtract_sky.flux_sky");
  cpl_ensure_code(p, CPL_ERROR_DATA_NOT_FOUND);
  aParams->flux_sky = cpl_parameter_get_double(p);
      
  p = cpl_parameterlist_find(aParameters, "muse.muse_scipost_subtract_sky.flux_lamp");
  cpl_ensure_code(p, CPL_ERROR_DATA_NOT_FOUND);
  aParams->flux_lamp = cpl_parameter_get_double(p);
    
  return 0;
} /* muse_scipost_subtract_sky_params_fill() */

/*----------------------------------------------------------------------------*/
/**
  @private
  @brief    Execute the plugin instance given by the interface
  @param    aPlugin   the plugin
  @return   0 if everything is ok, -1 if not called as part of a recipe
 */
/*----------------------------------------------------------------------------*/
static int
muse_scipost_subtract_sky_exec(cpl_plugin *aPlugin)
{
  if (cpl_plugin_get_type(aPlugin) != CPL_PLUGIN_TYPE_RECIPE) {
    return -1;
  }
  muse_processing_recipeinfo(aPlugin);
  cpl_recipe *recipe = (cpl_recipe *)aPlugin;
  cpl_msg_set_threadid_on();

  cpl_frameset *usedframes = cpl_frameset_new(),
               *outframes = cpl_frameset_new();
  muse_scipost_subtract_sky_params_t params;
  muse_scipost_subtract_sky_params_fill(&params, recipe->parameters);

  cpl_errorstate prestate = cpl_errorstate_get();

  muse_processing *proc = muse_processing_new("muse_scipost_subtract_sky",
                                              recipe);
  int rc = muse_scipost_subtract_sky_compute(proc, &params);
  cpl_frameset_join(usedframes, proc->usedframes);
  cpl_frameset_join(outframes, proc->outframes);
  muse_processing_delete(proc);
  
  if (!cpl_errorstate_is_equal(prestate)) {
    /* dump all errors from this recipe in chronological order */
    cpl_errorstate_dump(prestate, CPL_FALSE, muse_cplerrorstate_dump_some);
    /* reset message level to not get the same errors displayed again by esorex */
    cpl_msg_set_level(CPL_MSG_INFO);
  }
  /* clean up duplicates in framesets of used and output frames */
  muse_cplframeset_erase_duplicate(usedframes);
  muse_cplframeset_erase_duplicate(outframes);

  /* to get esorex to see our classification (frame groups etc.), *
   * replace the original frameset with the list of used frames   *
   * before appending product output frames                       */
  /* keep the same pointer, so just erase all frames, not delete the frameset */
  muse_cplframeset_erase_all(recipe->frames);
  cpl_frameset_join(recipe->frames, usedframes);
  cpl_frameset_join(recipe->frames, outframes);
  cpl_frameset_delete(usedframes);
  cpl_frameset_delete(outframes);
  return rc;
} /* muse_scipost_subtract_sky_exec() */

/*----------------------------------------------------------------------------*/
/**
  @private
  @brief    Destroy what has been created by the 'create' function
  @param    aPlugin   the plugin
  @return   0 if everything is ok, -1 if not called as part of a recipe
 */
/*----------------------------------------------------------------------------*/
static int
muse_scipost_subtract_sky_destroy(cpl_plugin *aPlugin)
{
  /* Get the recipe from the plugin */
  cpl_recipe *recipe;
  if (cpl_plugin_get_type(aPlugin) == CPL_PLUGIN_TYPE_RECIPE) {
    recipe = (cpl_recipe *)aPlugin;
  } else {
    return -1;
  }

  /* Clean up */
  cpl_parameterlist_delete(recipe->parameters);
  muse_processinginfo_delete(recipe);
  return 0;
} /* muse_scipost_subtract_sky_destroy() */

/*----------------------------------------------------------------------------*/
/**
  @private
  @brief    Add this recipe to the list of available plugins.
  @param    aList   the plugin list
  @return   0 if everything is ok, -1 otherwise (but this cannot happen)

  Create the recipe instance and make it available to the application using the
  interface. This function is exported.
 */
/*----------------------------------------------------------------------------*/
int
cpl_plugin_get_info(cpl_pluginlist *aList)
{
  cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe);
  cpl_plugin *plugin = &recipe->interface;

  char *helptext;
  if (muse_cplframework() == MUSE_CPLFRAMEWORK_ESOREX) {
    helptext = cpl_sprintf("%s%s", muse_scipost_subtract_sky_help,
                           muse_scipost_subtract_sky_help_esorex);
  } else {
    helptext = cpl_sprintf("%s", muse_scipost_subtract_sky_help);
  }

  /* Initialize the CPL plugin stuff for this module */
  cpl_plugin_init(plugin, CPL_PLUGIN_API, MUSE_BINARY_VERSION,
                  CPL_PLUGIN_TYPE_RECIPE,
                  "muse_scipost_subtract_sky",
                  "Subtract night sky model.",
                  helptext,
                  "Ole Streicher",
                  "usd-help@eso.org",
                  muse_get_license(),
                  muse_scipost_subtract_sky_create,
                  muse_scipost_subtract_sky_exec,
                  muse_scipost_subtract_sky_destroy);
  cpl_pluginlist_append(aList, plugin);
  cpl_free(helptext);

  return 0;
} /* cpl_plugin_get_info() */

/**@}*/