﻿' Licensed to the .NET Foundation under one or more agreements.
' The .NET Foundation licenses this file to you under the MIT license.
' See the LICENSE file in the project root for more information.

Imports System.Threading.Tasks
Imports Microsoft.CodeAnalysis.Remote.Testing

Namespace Microsoft.CodeAnalysis.Editor.UnitTests.FindReferences
    Partial Public Class FindReferencesTests
        <WorkItem(539174, "http://vstfdevdiv:8080/DevDiv2/DevDiv/_workitems/edit/539174")>
        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestVisualBasic_OperatorError1(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="Visual Basic" CommonReferences="true">
        <Document>
Module Program
    Sub Main(args As String())
        Dim b = 5 $$-
    End Sub
End Module
        </Document>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestCSharpFindReferencesOnUnaryOperatorOverload(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document>
class A
{
    void Goo()
    {
        A a;
        var x = $$[|-|]a;
    }
    public static A operator {|Definition:-|}(A a) { return a; }}
}
        </Document>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestCSharpFindReferencesOnUnaryOperatorOverloadFromDefinition(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document>
class A
{
    void Goo()
    {
        A a;
        var x = [|-|]a;
    }
    public static A operator {|Definition:$$-|}(A a) { return a; }
}
        </Document>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestCSharpFindReferencesOnBinaryOperatorOverload(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document>
class A
{
    void Goo()
    {
        var x = new A() [|$$+|] new A();
    }
    public static A operator {|Definition:+|}(A a, A b) { return a; }
}
        </Document>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestCSharpFindReferencesOnBinaryOperatorOverloadFromDefinition(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document>
class A
{
    void Goo()
    {
        var x = new A() [|+|] new A();
    }
    public static A operator {|Definition:$$+|}(A a, A b) { return a; }
}
        </Document>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        <WorkItem(52654, "https://github.com/dotnet/roslyn/issues/52654")>
        Public Async Function TestCSharpFindReferencesOnEqualsOperator(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document><![CDATA[
class A
{
    void Goo()
    {
        var x = new A() [|==|] new A();
    }
    public static bool operator {|Definition:$$==|}(A left, A right) => throw new System.NotImplementedException();
    public static bool operator !=(A left, A right) => throw new System.NotImplementedException();
}]]>
        </Document>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        <WorkItem(52654, "https://github.com/dotnet/roslyn/issues/52654")>
        Public Async Function TestCSharpFindReferencesOnNotEqualsOperator(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document><![CDATA[
class A
{
    void Goo()
    {
        var x = new A() [|!=|] new A();
    }
    public static bool operator ==(A left, A right) => throw new System.NotImplementedException();
    public static bool operator {|Definition:$$!=|}(A left, A right) => throw new System.NotImplementedException();
}]]>
        </Document>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        <WorkItem(52654, "https://github.com/dotnet/roslyn/issues/52654")>
        Public Async Function TestCSharpFindReferencesOnGreaterThanOperator(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document><![CDATA[
class A
{
    void Goo()
    {
        var x = new A() [|>|] new A();
    }
    public static bool operator {|Definition:$$>|}(A left, A right) => throw new System.NotImplementedException();
    public static bool operator <(A left, A right) => throw new System.NotImplementedException();
}]]>
        </Document>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        <WorkItem(52654, "https://github.com/dotnet/roslyn/issues/52654")>
        Public Async Function TestCSharpFindReferencesOnLessThanOperator(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document><![CDATA[
class A
{
    void Goo()
    {
        var x = new A() [|<|] new A();
    }
    public static bool operator >(A left, A right) => throw new System.NotImplementedException();
    public static bool operator {|Definition:$$<|}(A left, A right) => throw new System.NotImplementedException();
}]]>
        </Document>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        <WorkItem(52654, "https://github.com/dotnet/roslyn/issues/52654")>
        Public Async Function TestCSharpFindReferencesOnGreaterThanOrEqualsOperator(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document><![CDATA[
class A
{
    void Goo()
    {
        var x = new A() [|>=|] new A();
    }
    public static bool operator {|Definition:$$>=|}(A left, A right) => throw new System.NotImplementedException();
    public static bool operator <=(A left, A right) => throw new System.NotImplementedException();
}]]>
        </Document>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        <WorkItem(52654, "https://github.com/dotnet/roslyn/issues/52654")>
        Public Async Function TestCSharpFindReferencesOnLessThanOrEqualsOperator(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document><![CDATA[
class A
{
    void Goo()
    {
        var x = new A() [|<=|] new A();
    }
    public static bool operator >=(A left, A right) => throw new System.NotImplementedException();
    public static bool operator {|Definition:$$<=|}(A left, A right) => throw new System.NotImplementedException();
}]]>
        </Document>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestVisualBasicFindReferencesOnUnaryOperatorOverload(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="Visual Basic" CommonReferences="true">
        <Document>
Class A
    Public Shared Operator {|Definition:-|}(x As A) As A
        Return x
    End Operator

    Sub Goo()
        Dim a As A
        Dim b = $$[|-|]a
    End Sub
End Class
        </Document>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestVisualBasicFindReferencesOnUnaryOperatorOverloadFromDefinition(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="Visual Basic" CommonReferences="true">
        <Document>
Class A
    Public Shared Operator {|Definition:$$-|}(x As A) As A
        Return x
    End Operator

    Sub Goo()
        Dim a As A
        Dim b = [|-|]a
    End Sub
End Class
        </Document>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestVisualBasicFindReferencesOnBinaryOperatorOverload(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="Visual Basic" CommonReferences="true">
        <Document>
Class A
    Public Shared Operator {|Definition:^|}(x As A, y As A) As A
        Return y
    End Operator

    Sub Goo()
        Dim a = New A [|^$$|] New A
    End Sub
End Class
        </Document>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestVisualBasicFindReferencesOnBinaryOperatorOverloadFromDefinition(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="Visual Basic" CommonReferences="true">
        <Document>
Class A
    Public Shared Operator {|Definition:$$^|}(x As A, y As A) As A
        Return y
    End Operator

    Sub Goo()
        Dim a = New A [|^|] New A
    End Sub
End Class
        </Document>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WorkItem(30642, "https://github.com/dotnet/roslyn/issues/30642")>
        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestCSharpFindReferencesOnBuiltInOperatorWithUserDefinedEquivalent(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document>
class A
{
    void Goo(string a, string b, int x, int y)
    {
        var m = a $$[|==|] b;
        var n = a [|==|] b;
        var o = x == y;
    }
}
        </Document>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WorkItem(30642, "https://github.com/dotnet/roslyn/issues/30642")>
        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestVisualBasicFindReferencesOnBuiltInOperatorWithUserDefinedEquivalent(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="Visual Basic" CommonReferences="true">
        <Document>
class A
    sub Goo(a as string, b as string, x as integer, y as integer)
        dim m = a $$[|=|] b
        dim n = a [|=|] b
        dim o = x = y
    end sub
end class
        </Document>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WorkItem(30642, "https://github.com/dotnet/roslyn/issues/30642")>
        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestCrossLanguageFindReferencesOnBuiltInOperatorWithUserDefinedEquivalent_FromCSharp(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document>
class A
{
    void Goo(string a, string b, int x, int y)
    {
        var m = a $$[|==|] b;
        var n = a [|==|] b;
        var o = x == y;
    }
}
        </Document>
    </Project>
    <Project Language="Visual Basic" CommonReferences="true">
        <Document>
class A
    sub Goo(a as string, b as string, x as integer, y as integer)
        dim m = a [|=|] b
        dim n = a [|=|] b
        dim o = x = y
    end sub
end class
        </Document>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WorkItem(30642, "https://github.com/dotnet/roslyn/issues/30642")>
        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestCrossLanguageFindReferencesOnBuiltInOperatorWithUserDefinedEquivalent_FromVisualBasic(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document>
class A
{
    void Goo(string a, string b, int x, int y)
    {
        var m = a [|==|] b;
        var n = a [|==|] b;
        var o = x == y;
    }
}
        </Document>
    </Project>
    <Project Language="Visual Basic" CommonReferences="true">
        <Document>
class A
    sub Goo(a as string, b as string, x as integer, y as integer)
        dim m = a $$[|=|] b
        dim n = a [|=|] b
        dim o = x = y
    end sub
end class
        </Document>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WorkItem(44288, "https://github.com/dotnet/roslyn/issues/44288")>
        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestOperatorReferenceInGlobalSuppression(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document>
[assembly: System.Diagnostics.CodeAnalysis.SuppressMessage("Category", "RuleId", Scope = "member", Target = "~M:A.[|op_Addition|](A,A)~A")]

class A
{
    void Goo()
    {
        var x = new A() [|$$+|] new A();
    }

    public static A operator {|Definition:+|}(A a, A b) { return a; }
}
        </Document>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestCSharpFindOperatorUsedInSourceGeneratedDocument(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document>
class A
{
    public static A operator {|Definition:$$-|}(A a) { return a; }
}
        </Document>
        <DocumentFromSourceGenerator>
class B
{
    void Goo()
    {
        A a;
        var x = [|-|]a;
    }
}
        </DocumentFromSourceGenerator>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestCSharpStaticAbstractConversionOperatorInInterface(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document>
            <![CDATA[
interface I5<T> where T : I5<T>
{
    abstract static implicit operator {|Definition:i$$nt|}(T x);
}

class C5_1 : I5<C5_1>
{
    public static implicit operator {|Definition:int|}(C5_1 x) => default;
}

class C5_2 : I5<C5_2>
{
    static implicit I5<C5_2>.operator {|Definition:int|}(C5_2 x) => default;
}]]>
        </Document>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestCSharpStaticAbstractConversionOperatorViaFeature1(host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document>
            <![CDATA[
interface I5<T> where T : I5<T>
{
    abstract static implicit operator {|Definition:int|}(T x);
}

class C5_1 : I5<C5_1>
{
    public static implicit operator {|Definition:i$$nt|}(C5_1 x) => default;
}

class C5_2 : I5<C5_2>
{
    static implicit I5<C5_2>.operator int(C5_2 x) => default;
}]]>
        </Document>
    </Project>
</Workspace>
            Await TestStreamingFeature(input, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestCSharpStaticAbstractConversionOperatorViaFeature2(host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document>
            <![CDATA[
interface I5<T> where T : I5<T>
{
    abstract static implicit operator {|Definition:int|}(T x);
}

class C5_1 : I5<C5_1>
{
    public static implicit operator int(C5_1 x) => default;
}

class C5_2 : I5<C5_2>
{
    static implicit I5<C5_2>.operator {|Definition:i$$nt|}(C5_2 x) => default;
}]]>
        </Document>
    </Project>
</Workspace>
            Await TestStreamingFeature(input, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestCSharpStaticAbstractConversionOperatorViaApi1(host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document>
            <![CDATA[
interface I5<T> where T : I5<T>
{
    abstract static implicit operator {|Definition:int|}(T x);
}

class C5_1 : I5<C5_1>
{
    public static implicit operator {|Definition:int|}(C5_1 x) => default;
}

class C5_2 : I5<C5_2>
{
    static implicit I5<C5_2>.operator {|Definition:i$$nt|}(C5_2 x) => default;
}]]>
        </Document>
    </Project>
</Workspace>
            Await TestAPI(input, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestCSharpStaticAbstractConversionOperatorViaApi2(host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document>
            <![CDATA[
interface I5<T> where T : I5<T>
{
    abstract static implicit operator {|Definition:int|}(T x);
}

class C5_1 : I5<C5_1>
{
    public static implicit operator {|Definition:in$$t|}(C5_1 x) => default;
}

class C5_2 : I5<C5_2>
{
    static implicit I5<C5_2>.operator {|Definition:int|}(C5_2 x) => default;
}]]>
        </Document>
    </Project>
</Workspace>
            Await TestAPI(input, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestCSharpStaticAbstractOperatorInInterface(kind As TestKind, host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document>
            <![CDATA[
interface I4<T> where T : I4<T>
{
    abstract static int operator {|Definition:+$$|}(T x);
}

class C4_1 : I4<C4_1>
{
    public static int operator {|Definition:+|}(C4_1 x) => default;
}

class C4_2 : I4<C4_2>
{
    static int I4<C4_2>.operator {|Definition:+|}(C4_2 x) => default;
}]]>
        </Document>
    </Project>
</Workspace>
            Await TestAPIAndFeature(input, kind, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestCSharpStaticAbstractOperatorViaApi1(host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document>
            <![CDATA[
interface I4<T> where T : I4<T>
{
    abstract static int operator {|Definition:+|}(T x);
}

class C4_1 : I4<C4_1>
{
    public static int operator {|Definition:$$+|}(C4_1 x) => default;
}

class C4_2 : I4<C4_2>
{
    static int I4<C4_2>.operator {|Definition:+|}(C4_2 x) => default;
}]]>
        </Document>
    </Project>
</Workspace>
            Await TestAPI(input, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestCSharpStaticAbstractOperatorViaApi2(host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document>
            <![CDATA[
interface I4<T> where T : I4<T>
{
    abstract static int operator {|Definition:+|}(T x);
}

class C4_1 : I4<C4_1>
{
    public static int operator {|Definition:+|}(C4_1 x) => default;
}

class C4_2 : I4<C4_2>
{
    static int I4<C4_2>.operator {|Definition:$$+|}(C4_2 x) => default;
}]]>
        </Document>
    </Project>
</Workspace>
            Await TestAPI(input, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestCSharpStaticAbstractOperatorViaFeature1(host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document>
            <![CDATA[
interface I4<T> where T : I4<T>
{
    abstract static int operator {|Definition:+|}(T x);
}

class C4_1 : I4<C4_1>
{
    public static int operator {|Definition:$$+|}(C4_1 x) => default;
}

class C4_2 : I4<C4_2>
{
    static int I4<C4_2>.operator +(C4_2 x) => default;
}]]>
        </Document>
    </Project>
</Workspace>
            Await TestStreamingFeature(input, host)
        End Function

        <WpfTheory, CombinatorialData, Trait(Traits.Feature, Traits.Features.FindReferences)>
        Public Async Function TestCSharpStaticAbstractOperatorViaFeature2(host As TestHost) As Task
            Dim input =
<Workspace>
    <Project Language="C#" CommonReferences="true">
        <Document>
            <![CDATA[
interface I4<T> where T : I4<T>
{
    abstract static int operator {|Definition:+|}(T x);
}

class C4_1 : I4<C4_1>
{
    public static int operator +(C4_1 x) => default;
}

class C4_2 : I4<C4_2>
{
    static int I4<C4_2>.operator {|Definition:$$+|}(C4_2 x) => default;
}]]>
        </Document>
    </Project>
</Workspace>
            Await TestStreamingFeature(input, host)
        End Function
    End Class
End Namespace
