---
layout: default
title: PSR-7 compliant URI Object API
---

PSR interoperability
=======

As we are dealing with URI, the package provides classes compliant
with [PSR-7](https://www.php-fig.org/psr/psr-7/) and 
[PSR-17](https://www.php-fig.org/psr/psr-17/). This is done to allow
more interoperability amongs PHP packages.

## PSR-7 compatibility

The `Http` class implements the PSR-7 `UriInterface` interface. This means that you can
use this class anytime you need a PSR-7 compliant URI object. Behind the scene, the
implementation uses the [Uri object](/uri/7.0/rfc3986/) and thus presents the same
features around URI validation, modification and normalization.

<p class="message-notice">The class handles all URI schemes <strong>BUT</strong> default
to HTTP(s) rules if the scheme is not present and not recognized as special.</p>

While the default constructor is private and can not be accessed to instantiate a new object,
the `League\Uri\Http` class comes with named constructors to ease instantiation.

The following examples show how to use the different named constructors:

~~~php
<?php

use League\Uri\Http;
use League\Uri\UriString;
use Laminas\Diactoros\Uri as LaminasUri;

// using a string or an object which expose the `__toString` method

$uri = Http::new('http://example.com/path/to?q=foo%20bar#section-42');
echo $uri; // display 'http://example.com/path/to?q=foo%20bar#section-4'

$laminasUri = new LaminasUri("http://www.example.com/path/to/the/sky");
$laminasUri->getQuery(); //return '';

Http::new($laminasUri)->getQuery(); //return '';

// using `parse_url` or the package `UriString::parse` static method.

$uri = Http::fromComponents(UriString::parse("http://uri.thephpleague/7.0/uri/api"));

//don't forget to provide the $_SERVER array
$uri = Http::fromServer($_SERVER);
~~~

<p class="message-warning">If you supply your own hash to <code>fromComponents</code>,
you are responsible for providing well parsed components without their URI delimiters.</p>

<p class="message-warning">The <code>fromServer</code> method only relies on the server's 
safe parameters to determine the current URI. If you are using the library behind a 
proxy the result may differ from your expectation as no <code>$_SERVER['HTTP_X_*']</code>
header is taken into account for security reasons.</p>

You can also return a URI based on standard specifications:

~~~php
<?php

use League\Uri\Http;

$uri = Http::fromBaseUri("./p#~toto", "http://www.example.com/path/to/the/sky/");
echo $uri; //displays "http://www.example.com/path/to/the/sky/p#~toto"

$template = 'https://example.com/hotels/{hotel}/bookings/{booking}';
$variables = ['booking' => '42', 'hotel' => 'Rest & Relax'];
echo Http::fromTemplate($template, $variables)->toString();
//displays "https://example.com/hotels/Rest%20%26%20Relax/bookings/42"
~~~

The `fromBaseUri` method resolves URI using the same logic behind URL construction
in a browser and [is inline with how the Javascript](https://developer.mozilla.org/en-US/docs/Web/API/URL/URL) `URL` object constructor works.
If no base URI is provided, the URI to resolve **MUST** be absolute. Otherwise, the base URI **MUST** be absolute.

The `fromTemplate` method resolves a URI using the rules and variable from the
[URITemplate specification RFC6570](http://tools.ietf.org/html/rfc6570):
The method expects at most two parameters. The URI template to resolve and the variables use
for resolution. You can get a more in-depth understanding of
[URI Template](/uri/7.0/uri-template) in its dedicated section of the documentation.

In addition to PSR-7 compliance, the class implements PHP's `JsonSerializable` interface.

~~~php
<?php
use League\Uri\Http;

echo json_encode(Http::new('http://example.com/path/to?q=foo%20bar#section-42'));
// display "http:\/\/example.com\/path\/to?q=foo%20bar#section-42"
~~~

## PSR-17 compatibility

The package also provides an implementation of the `UriFactoryInterface` from [PSR-17](https://www.php-fig.org/psr/psr-17/)

~~~php
<?php
use League\Uri\HttpFactory;

$uriFactory = new HttpFactory();
$uri = $uriFactory->createUri('http://example.com/path/to?q=foo%20bar#section-42');
echo $uri::class; // display League\Uri\Http
~~~
