#!/usr/bin/env python
# file: test_fastq_join.py

# Tests for the fastq_join.py application controller.
# Application controller for 'fastq-join' ea-utils v1.1.2-537 
# http://code.google.com/p/ea-utils/

from cogent.util.unit_test import TestCase, main
from cogent.util.misc import create_dir
from cogent.app.fastq_join import FastqJoin, join_paired_end_reads_fastqjoin
from cogent.app.util import get_tmp_filename
import os 
import shutil

__author__ = "Michael Robeson"
__copyright__ = "Copyright 2007-2013, The Cogent Project"
__credits__ = ["Michael Robeson"]
__license__ = "GPL"
__version__ = "1.9"
__maintainer__ = "Michael Robeson"
__email__ = "robesonms@ornl.gov"
__status__ = "Development"


class FastqJoinTests(TestCase):
    """Tests for fastq-join application controller."""

    def setUp(self):
        """General setup for fastq-join tests"""
        self.temp_dir_string = '/tmp/test_fastq_join/'
        create_dir(self.temp_dir_string)

        self.temp_dir_string_spaces = '/tmp/test fastq join/'
        create_dir(self.temp_dir_string_spaces)

        # create temp file path strings
        self.test_fn1 = os.path.join(self.temp_dir_string,'reads1.fastq')
        self.test_fn1_space = os.path.join(self.temp_dir_string_spaces, 
                                        'reads1.fastq')
        self.test_fn2 = os.path.join(self.temp_dir_string,'reads2.fastq')
        self.test_fn2_space = os.path.join(self.temp_dir_string_spaces,
                                        'reads2.fastq')

    def writeTmpFastq(self, fw_reads_path, rev_reads_path):
        """write forward and reverse reads data to temp fastq files"""
        try:
            fq1 = open(fw_reads_path, "w+")
            fq1.write(reads1_string)
            fq1.close()
            fq2 = open(rev_reads_path, "w+")
            fq2.write(reads2_string)
            fq2.close()
        except OSError:
            pass
   
    def test_base_command(self):
        """fastq-join command should return correct BaseCommand"""
        c = FastqJoin()
        # test base command 
        self.assertEqual(c.BaseCommand, ''.join(['cd "', os.getcwd(), '/"; ',
                                                  'fastq-join']))
        # test turning on a parameter
        c.Parameters['-p'].on('15')
        self.assertEqual(c.BaseCommand, ''.join(['cd "', os.getcwd(), '/"; ',
                                                 'fastq-join -p 15'])) 

    def test_changing_working_dir(self):
        """WorkingDir should change properly"""
        c = FastqJoin(WorkingDir=self.temp_dir_string)
        self.assertEqual(c.BaseCommand, 
                         ''.join(['cd "', self.temp_dir_string,
                                  '/"; ', 'fastq-join']))
        c = FastqJoin()
        c.WorkingDir = self.temp_dir_string + '2'
        self.assertEqual(c.BaseCommand, 
                         ''.join(['cd "', self.temp_dir_string + '2', 
                                  '/"; ', 'fastq-join']))

    def test_fastqjoin_assembly(self):
        """Runs FastqJoin with default and alternate settings.

        Checks ouput of assembled paired-ends, unassebled files, 
        and reports file.  
        """
        self.writeTmpFastq(self.test_fn1, self.test_fn2)
        
        ### Default settings
        params={}
        params['-p'] = 8
        params['-m'] = 6
        params['-o'] = 'test_fastq'
        params['-r'] = 'test_fastq_report'
 
        fastqjoin_app = FastqJoin(params=params,
                                  WorkingDir=self.temp_dir_string)

        res = fastqjoin_app([self.test_fn1, self.test_fn2])
        
        # test if file strings are valid:
        self.assertEqual(res['Assembled'].read(), expected_assembly)
        self.assertEqual(res['UnassembledReads1'].read(), expected_default_un1)
        self.assertEqual(res['UnassembledReads2'].read(), expected_default_un2)
        self.assertEqual(res['Report'].read(), expected_report)
        
        res.cleanUp()

        ### Alternate settings
        params={}
        params['-p'] = 5
        params['-m'] = 10
        params['-o'] = 'test_fastq'
        working_dir=self.temp_dir_string
 
        fastqjoin_app_alt = FastqJoin(params=params, WorkingDir=working_dir)

        res_alt = fastqjoin_app_alt([self.test_fn1, self.test_fn2])
        
        # test if file strings are valid:
        self.assertEqual(res_alt['Assembled'].read(), 
                         expected_assembly_alt_param)
        self.assertEqual(res_alt['UnassembledReads1'].read(), 
                         expected_default_un1_alt_param)
        self.assertEqual(res_alt['UnassembledReads2'].read(), 
                         expected_default_un2_alt_param)
        
        res_alt.cleanUp()
        shutil.rmtree(self.temp_dir_string)


    def test_join_paired_end_reads_fastqjoin(self):
        """join_paired_end_reads_fastqjoin: should work as expected """
        self.writeTmpFastq(self.test_fn1, self.test_fn2)

        ### 1: run with default function params ###
        jp_file_path = join_paired_end_reads_fastqjoin(self.test_fn1, self.test_fn2,
                                     working_dir=self.temp_dir_string)
        
        # test if joined paired end output is correct:
        jp_ends = open(jp_file_path['Assembled'],'U').read()
        self.assertEqual(jp_ends, expected_assembly)
       
        jp_unass_1 = open(jp_file_path['UnassembledReads1'],'U').read()
        self.assertEqual(jp_unass_1, expected_default_un1)

        jp_unass_2 = open(jp_file_path['UnassembledReads2'],'U').read()
        self.assertEqual(jp_unass_2, expected_default_un2)


        ### 2: test with different parameters ###
        jp_file_path_alt = join_paired_end_reads_fastqjoin(self.test_fn1, self.test_fn2,
                                         perc_max_diff=5, 
                                         min_overlap=10, 
                                         working_dir=self.temp_dir_string)

        # test if joined paired end alt output is correct:
        jp_ends_b = open(jp_file_path_alt['Assembled'],'U').read()
        self.assertEqual(jp_ends_b, expected_assembly_alt_param)
       
        jp_unass_1b = open(jp_file_path_alt['UnassembledReads1'],'U').read()
        self.assertEqual(jp_unass_1b, expected_default_un1_alt_param)

        jp_unass_2b = open(jp_file_path_alt['UnassembledReads2'],'U').read()
        self.assertEqual(jp_unass_2b, expected_default_un2_alt_param)

        
        ### test for None as input to function:
        # output should be identical to default (v1.1.2-537)
        jp_file_path_none = join_paired_end_reads_fastqjoin(self.test_fn1, self.test_fn2,
                                     perc_max_diff=None,
                                     min_overlap=None,
                                     working_dir=self.temp_dir_string)
        
        # test if joined paired end output is correct:
        jp_ends_none = open(jp_file_path_none['Assembled'],'U').read()
        self.assertEqual(jp_ends, expected_assembly)
       
        jp_unass_1_none = open(jp_file_path_none['UnassembledReads1'],'U').read()
        self.assertEqual(jp_unass_1, expected_default_un1)

        jp_unass_2_none = open(jp_file_path_none['UnassembledReads2'],'U').read()
        self.assertEqual(jp_unass_2, expected_default_un2)

        # final cleanup
        os.remove(self.test_fn1) 
        os.remove(self.test_fn2)
        shutil.rmtree(self.temp_dir_string) 



# test strings

reads1_string ="""@MISEQ03:123:000000000-A40KM:1:1101:14149:1572 1:N:0:TCCACAGGAGT
GGGGGGTGCCAGCCGCCGCGGTAATACGGGGGGGGCAAGCGTTGTTCGGAATTACTGGGCGTAAAGGGCTCGTAGGCGGCCCACTAAGTCAGACGTGAAATCCCTCGGCTTAACCGGGGAACTGCGTCTGATACTGGATGGCTTGAGGTTGGGAGAGGGATGCGGAATTCCAGGTGTAGCGGTGAAATGCGTAGATATCTGGAGGAACACCGGTGGCGAAGGCGGCATCCTGGACCAATTCTGACGCTGAG
+
CCCCCCCCCFFFGGGGGGGGGGGGHHHHGGGGGFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF-.;FFFFFFFFF9@EFFFFFFFFFFFFFFFFFFF9CFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFBFFFFFFFFFFFFFECFFFEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFBFFFFFFFFFFFFFFFFF;CDEA@FFFFFFFFFFFFFFFFFFFFFFFFFF
@MISEQ03:123:000000000-A40KM:1:1101:14170:1596 1:N:0:TCCACAGGAGT
ATGGCGTGCCAGCAGCCGCGGTAATACGGAGGGGGCTAGCGTTGTTCGGAATTACTGGGCGTAAAGCGCACGTAGGCGGCTTTGTAAGTTAGAGGTGAAAGCCCGGGGCTCAACTCCGGAACTGCCTTTAAGACTGCATCGCTAGAATTGTGGAGAGGTGAGTGGAATTCCGAGTGTAGAGGTGAAATTCGTAGATATTCGGAAGAACACCAGTGGCGAAGGCGACTCACTGGACACATATTGACGCTGAG
+
CCCCCCCCCCFFGGGGGGGGGGGGGHHHGGGGGGGGGHHHGGGGGHHGGGGGHHHHHHHHGGGGHHHGGGGGHHHGHGGGGGHHHHHHHHGHGHHHHHHGHHHHGGGGGGHHHHHHHGGGGGHHHHHHHHHGGFGGGGGGGGGGGGGGGGGGGGGGGGFGGFFFFFFFFFFFFFFFFFF0BFFFFFFFFFFFFEFFFFFFFFFFFFFFFFFFFFFFFFFFFDFAFCFFFFFFFFFFFFFFFFFFBDFFFFF
@MISEQ03:123:000000000-A40KM:1:1101:14740:1607 1:N:0:TCCACAGGAGT
AGTGTGTGCCAGCAGCCGCGGTAATACGTAGGGTGCGAGCGTTAATCGGAATTACTGGGCGTAAAGCGTGCGCAGGCGGTTCGTTGTGTCTGCTGTGAAATCCCCGGGCTCAACCTGGGAATGGCAGTGGAAACTGGCGAGCTTGAGTGTGGCAGAGGGGGGGGGAATTCCGCGTGTAGCAGTGAAATGCGTAGAGATGCGGAGGAACACCGATGGCGAAGGCAACCCCCTGGGATAATATTTACGCTCAT
+
AABCCFFFFFFFGGGGGGGGGGGGHHHHHHHEGGFG2EEGGGGGGHHGGGGGHGHHHHHHGGGGHHHGGGGGGGGGGGGEGGGGHEG?GBGGFHFGFFHHGHHHGGGGCCHHHHHFCGG01GGHGHGGGEFHH/DDHFCCGCGHGAF;B0;DGF9AEEGGGF-=C;.FFFF/.-@B9BFB/BB/;BFBB/..9=.9//:/:@---./.BBD-@CFD/=A-::.9AFFFFFCEFF./FBB############
@MISEQ03:123:000000000-A40KM:1:1101:14875:1613 1:N:0:TCCACAGGAGT
GGTGGGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAG
+
CCCCCCCCCFFFGGGGGGGGGGGGGHHHHHHGGGGGHHHHGGGGGHHGGGGGHHHHHHHHGGGGHHHGGGGGGGGGHHGGGHGHHHHHHHHHHHHHHHHHGHHHGGGGGGHHHHHHHHGGHGGHHGHHHHHHHHHFHHHHHHHHHHHHHHHGHHHGHHGEGGDGGFFFGGGFGGGGGGGGGGGFFFFFFFDFFFAFFFFFFFFFFFFFFFFFFFFFFFFFFDFFFFFFFEFFFFFFFFFB:FFFFFFFFFF
@MISEQ03:123:000000000-A40KM:1:1101:14902:1627 1:N:0:TCCACAGGAGT
AGGGGGTGCCAGCCGCCGCGGTAAGACGAAGGTGGCTAGCGTTATTCGGATTTACTGGGCATAAAGCGCGCGTAGGCGGCTTGGCAAGTCGGAGGTGAAAGCCTGCAGCTCAACTGCGGTGTGCCTCCGAAACTGCCGAGCTTGAGTGCGGGAGGGGTGATTGGAACTCCAGGTGTAGCGGTGAAATGCGTAGATATCTGGAGGAACACCAGAGGCGAAGGCGAATCACTGGACCGCAACTGACGCTGAGG
+
ABBBBBBBBFFCFGGGGGGGGGGGHHHGGGGGHHGHHHHHGGGGHHHHHGEGHHHHHHHHHHHHHHHGGGGGGGGGHGGGGGGGHFHGHHGGGGGGHHHHHHFHHHHHHHHHHHHCGFGGCGGHHHHGGGDGDGFGGCFGFDGGGGCFFFFF-@DAFDDEFFFFFFFFFFFFFFFFFFFFF9-9BBFFFFFFAFFFFFFFFBF??FFFF?AAFEEAAFDFFFFFFFFFFB/BBFFFFFDFF//BADDD..E
@MISEQ03:123:000000000-A40KM:1:1101:15883:1644 1:N:0:TCCACAGGAGT
GGGATGTGCCAGCAGCCGCGGTAATACAGAGGTCCCGAGTGTTGTTCGGATTCACTGGGCGTAAAGGGTGCGTAGGTGGTGGGGTAAGTCTCGTGTGAAAGCTCGCAGCTTAACTGCGAAAAGTCACGGGATACTATTCCGCTGGAGGCTTGGAGGGGGGACTGGAATACTTGGTGTAGCAGTGAAATGCGTAGATATCAAGTGGAACACCAGTGGCGAAGGCGAGTCCCTGGACAAGACCTGACACTGAG
+
CDDDDDFFFFFFGGGGGGGGGGGGHHHHHHHGHGGH2EFGHHHHHHHGGGGGHHHHHHHHGGGGHHHGHGGGGGGGHHGGGGGGGGHHHHHHHGHHHHHHHHHGGGGGGHHHHHHDGGGGGGHHHHHGGGGGHHHHHHHHGGGGGGHGGHHGGGHGGGGFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFD;BFFFFFFFFFFFFFFFFFFFFFFFFFAFDEFF@DFFFFFFEFFFFFFFFF9;B;FFFF
@MISEQ03:123:000000000-A40KM:1:1101:17346:1668 1:N:0:TCCACAGGAGT
GGCGGGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAACGTTGCTCGGAATCACTGGGCGTAAAGGGCGTGTAGGCGGGAAGGAAAGTCGGGCGTGAAATCCCTGGGCTCAACCCAGGAACTGCACTCGAAACTTCCTTTCTTGAGTGCCGGAGAGGAAAGCGGAATTCCTGGTGTAGAGGTGAAATTCGTAGATATCAGGAGGAACACCAGTGGCGAAGGCGGCTTTCTGGGCGGTAAATGACGCTGAG
+
BCCCCDDBCFFFGGGGGGGGGGGGHHHHHGGGGAFEHFHHGHHHCFHGGEGGHHHGHGHHGGGGHHHGEEEGEFHFFGGG@/?C?GGF>FDGFGGCGDGHFHHHHGHGGHCGHHH?.CE.;A0FF0FFFEAFG?CFBFFGGFFF9;0;BFFFA:@>DFFFFAFDF-CEFFFFFFFF9/FBFF:AFFFFF9EBFEE/:FFFFF/9A.;.:;DEFFFFFFF;DFE=D9-;;/;BB/.:DDFF###########
@MISEQ03:123:000000000-A40KM:1:1101:14457:1707 1:N:0:TCCACAGGAGT
TGGGTGTGCCAGCCGCCGCGGTAATACGAGGGGTGCAAGCGTTGCTCGGAATTATTGGGCGTAAAGGGTAGGTAGGTGGTCTCATTTGTCTGGGGTGAAAGCCTCGAGCTTAACTCGAGAAGTGCCCCAGAAACGGTGAGACTAGAGTTCTGGGGAGGGTCGTGGAATTCCCGGTGTAGCGGTGAAATGCGTAGAGATCGGGAGGAACACCAGAGGCGAAGGCGACCACCTGGACTGATACTGACACTGAG
+
AABCBCCCFFFFGGGGGGGGGGGGHHHHGGGGGGGGHHHHGGGGGHHGGFGGHHHHHHHHGGGGHHHGHHGGHHHGHHGGHH?4FHHHHHHHHHGGGHHHHHHHHGGGGGHHHHHHGHGGGHHHHHHGGGHHHHGGGHGHFHHHHHHHHHHHHHGGGGGGGGGGGGGGGGGGGGGGFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFF:;FFEFBFFFFF
@MISEQ03:123:000000000-A40KM:1:1101:15133:1720 1:N:0:TCCACAGGAGT
TATAGGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAG
+
DDDDDFFFFFFFGGGGGGGGGGGGHHHHHHHGGGGGHHHHGGGGGHHGGGGGHHHHHHHHGGGGHHHGGGGGGGGGHHGGGHGHHHHHHHHHHHHHHHHHHHHHGGGGGGHHHHHHHHGHHGGHHHHHHGHHHHHHHHHHHHHHHHHHHHHHHHHHHHGGGGGEGGGFGGGGGGGGGGGGGGGGGGGGGFFFFFEFFFFFFFFFFEFFFFFFFFFFFFFAFDEFFFFFFFFFFFFFFFFB;FFFFFFFFEF
@MISEQ03:123:000000000-A40KM:1:1101:16328:1722 1:N:0:TCCACAGGAGT
AGAGGGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAG
+
ABCCBBBCCFFBGGGGGGGGGGGGHHHHHGHGGGGGHHHHGGGGGHHGGGGGHHHHHHHHCGGGHHHGGGGGGGGGHHGGGHGHHHHGHHHHHHHHHHHHGHHHGGGGGGHHHHHHHHGGHGGGHFHHHGFHHHHHHHHHGHHHHHGHHFHHHHHHHGGCADGFGGGGGGGGGGGGGGGGGAFFFFFFFFFFDB.FFFFFFFFFFFFFFFFFFFFFFFFCFDECFFFFFFFFFFFFFFFBFFFFFFFFFFF
@MISEQ03:123:000000000-A40KM:1:1101:17115:1764 1:N:0:TCCACAGGAGT
GTGGGGTGCCAGCAGCCGCGGTAATACGGAGGATCCAAGCGTTATCCGGAATCATTGGGTTTAAAGGGTCCGTAGGCGGTTTAATAAGTCAGTGGTGAAAGCCCATCGCTCAACGGTGGAACGGCCATTGATACTGTTAAACTTGAATTATTAGGAAGTAACTAGAATATGTAGTGTAGCGGTGAAATGCTTAGAGATTACATGGAATACCAATTGCGAAGGCAGGTTACTACTAATGGATTGACGCTGAT
+
CCCCCCBBCFFFGGGGGGGGGGGGHHHHGGGGGHHHHHHGGGGGHHHGGGGGHHHHHHHGGGHHHHHGHHHGHHGGHGGGFGG4FFHHGHHHHHHHGHHHHHHHHHHGGHHGHHG/??DFG1<CDCGHHHGHGHHHHHHHGHHHHFHHHHHHBGHHHHFHGHGHHGHHGHHHHHHHGGGGGGGGGGGGGFFGGG099;CFFFCFGBCCFBFFFFFFFFFFFFFFFFFBFBFFFFFFFBFFFFFFFFFFFAF
@MISEQ03:123:000000000-A40KM:1:1101:14429:1791 1:N:0:TCCACAGGAGT
GTACCGTGCCAGCAGCCGCGGTAATACGTAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGTGCGCAGGCGGTTTTGTAAGACAGTTGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGTGACTGCATAGCTAGAGTACGGTAGAGGGGGATGGAATTCCGCGTGTAGCAGTGAAATGCGTAGATATGCGGAGGAACACCGATGGCGAAGGCAATCCCCTGGACCTGTACTGACGCTCAT
+
ABBCCCCCCCFFGGGGGGGGGGGGHHHHHHHGGHGGHHHHGGGGGHHGGGGGHHHHHHHHGGGGHHHGGGGGGGGGGGGEE/?0FDHFF2FBGFDF/GHHHFHHG@BGFFEGHFHFFHGAHFHFGGDGHBHBGHHHGHGHFHHHHHFBDGDDHHGHHHGGG-CEFBFFGFG9C?FGFEFEBFFFFFFFFBAA:DEEB99FADABF.F?FFF-BDDFFFDAAFAFFFFFFFFFFBFF9B#############
@MISEQ03:123:000000000-A40KM:1:1101:14969:1845 1:N:0:TCCACAGGAGT
CGCCGGGTGCCAGCCGCCGCGGTAATACGTAGGGGGCTAGCGTTGTCCGGATTTATTGGGCGTAAAGAGCGTGTAGGCGGCCAGGTAGGTCCGGTGTGAAAACTCGAGGCTTAACCTCGAGATGTCATCGGAAACCATCTGGCTAGAGTCCGGAAGAGGAGAGTGGAATTCCTGGTGTAGCGGTGAAATGCGCAGATATCAGGAAGAACACCTATGGCGAAAGCAGCTCTCTGGGACGGTACTGACGCTGA
+
AAABBB@ABBFFGGGGGGGGGFGGGGGGHHHGGGGGGFFHGGGGGGHEGGEGGHFHHHHHHGGGGHHHHHGGGGHHGGGCGGCFFHHHGHHHDCADGGHHFF<GHGHGGGHHHGHHHHGGGFGHHHHHHGGGGGGGGGGGGGGGGGGGGGGGGDGGFGFEBEFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFFFAFBFFFFFBFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFAFDFAEFBFBFADFF#
@MISEQ03:123:000000000-A40KM:1:1101:17499:1855 1:N:0:TCCACAGGAGT
AGCCGGTGCCAGCCGCCGCGGTAATACGGAGGATCCAAGCGTTATCCGGAATCATTGGGTTTAAAGGGTCCGTAGGCGGTTTAGTAAGTCAGTGGTGAAAGCCCATCGCTCAACGGTGGAACGGCCATTGATACTGCTAGACTTGAATTATTAGGAAGTAACTAGAATATGTAGTGTAGCGGTGAAATGCTTAGAGATTACATGGAATACCAATTGCGAAGGCAGGTTACTACTAATGGATTGACGCTGAT
+
CCCCCCCCBFFFGGGGGGGGGGGGHHHHGGGGGHHHHHHGGGGGHHHGGGGGHHHHHHHGGGHHHHGGHGGGHHGGHGGGGGHHHHHHHHHHHHHGGGHHHHHHHHHGGHHGHHG/??EGHHDEGGGHHHHHHHHHHHDGHHHHHGEHHHHHHHHHHHHCHCGHHHGFFGGGGGGGGGGGGGGGGGGGGGFGGG09;;CFFFFEFFBBBFFFFFFFFFFAF?EFFFFFF9FF0FFBFBFFFFFFBDFFFFF
@MISEQ03:123:000000000-A40KM:1:1101:14009:1889 1:N:0:TCCACAGGAGT
GCGGCGTGCCAGCCGCTGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAG
+
ABBBBBBBBBFFGGGGFGGGGGGGHHHHHHHGGGGGHHHHGGGGGHHGGGGGHGHHHHHHGGGGHHHGGGGGGGGGGHGGGHGHHHHHHHHHHHHHHHHHHHHHGGGGGGHHHHHHHHGFHGHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHGDGGGGGGGGGGGGGGGGGGGGGGGFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFBFFFFFFFFFFFF;BFFEFFFFFFF
@MISEQ03:123:000000000-A40KM:1:1101:17869:1908 1:N:0:TCCACAGGAGT
ATTGCGTGCCAGCCGCCGCGGTAATACAGAGGTCCCAAGCGTTGTTCGGATTCACTGGGCGTAAAGGGTGCGTAGGCGGTCGGGTAAGTCTGACGTGAAATCTCCGAGCCTAACTCGGAAACTGCGTCGGATACTATTCGGCTAGAGGAATGGAGGGGAGACTGGAATTCTCGGTGTAGCAGTGAAATGCGTAGATATCGAGAGGAACACCAGTGGCGAAGGCGAGTCTCTGGACATTTCCTGACGCTGAG
+
BCCCCCCCCCFFGGGGGGGGGGGGHHHHHHHGHHHHHHHGGGGGGHHGGGGGHHHHHHHHGGGGHHHGHGGGGGGGHGGGGGGGHGHHHHHHGHGGGHGHHHHHHGGGGGHHHHHFGGGGGHHHH<<<CFGFEGHBCHGHHGGGGGGGFGGGGGGDADDCFEFFFFFFFFFFFFFFFFFFBFFFFFFFFFFFFFFBFFFFEFFFFFFFFFFFFFFFFFF=FFFCADFEFFFFFBABBFFFFFFB:;DFDDF
@MISEQ03:123:000000000-A40KM:1:1101:14767:1933 1:N:0:TCCACAGGAGT
GCGTGGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGTGCGCAGGCGGTTTTGTAAGACAGTTGTGAAATCCCCGGGCTCAACCTGGGAACTGCCTTTAAAACTGCATCGCTTGAACATGGGAGAGGGGGGTGGAATTCCGAGTGTAGCGGTGAAATTCGTAGAGATTCGGAAGAACACCAGTGGCGAAGGCGGCCCCCTGGACTAATATTGACGCTGAG
+
3AAA?BBBFFFFGGGGGGGGGGGGHHHHHHHGGGGGHHHHGGGGGHHGGGGGHHHHHHHHGGGGHHHGFGGGGGGGGGGED??0GFGG1?BC1@FFDEHHHHHHGG@DCGHFHGHCGHG0GFHHHBHGG0G0=0G0<<00::C.:CF00;//.;EDGG.@-;@BBEFFF/B/.;;AFFBB9BABFFFFB9EFFFF/BF//99A.;.F.;FFE.9FFFBBBBDE9@B-9-;9EF.9/B/;/9;9B9.A####
@MISEQ03:123:000000000-A40KM:1:1101:17050:1939 1:N:0:TCCACAGGAGT
GGATGGTGCCAGCCGCCGCGGTAATACGTAGGTGGCAAGCGTTGTCCGGATTTATTGGGTTTAAAGGGTGCGTAGGCGGCCCTGTAAGTCAGTGCTGAAATATCCCGGCTTAACCGGGAGGGTGGCATTGATACTGCGGGGCTTGAGTACAGATGAAGTCGGCGGAATTGACGGTGTAGCGGTGAAATTCATAGATATCGGGAGGAACACCAGTGGCGAAGGCGAGTCTCTGGGCTGATACTGACGCTGAG
+
DDDCCCFFFFFFGGGGGGGGGGGGHHHHHHHGHHGHHHGHGGGGGHHGGEGGHHHHHHHGGGHHHHHGHGGGGGGGHGGGGGHHHHHHHHHHHHGHHHHGHHHFHHGGGGGHHHH<ACADGG<EGEHHHHHHHHHHHGGGGGGGGFGGGGGGGGGGGGGGGFCFFCCEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFDFFFFFFFFFFFFFAFDFFFFFFFFFFFFFFD?FFBFFFFFFFFFF
@MISEQ03:123:000000000-A40KM:1:1101:16782:1940 1:N:0:TCCACAGGAGT
GAGTTGTGCCAGCCGCCGCGGTAATACGTAGGGGGCAAGCGTTGTCCGGATTTATTGGGCGTAAAGAGCGTGTAGGCGGCTAGGTAGGTCCGTTGTGAAAACTCGAGGCTCAACCTCGAGACGTCGATGGAAACCATCTAGCTAGAGTCCGGAAGAGGAGAGTGGAATTCCTGGTGTAGCGGTGAAATGCGCAGATATCAGGAAGAACACCCGTGGCGAAGGCGGCTCTCTGGTACGTGACTGACGCTGAG
+
BCCCCFFCFFFFGGGGGGGGGGGGHGHHHHHGGGGGGGGHGGGGGHHGGGGGHHHHHHHHGGGGHHHHHGGGGHHHHGGGG/GHHHHGHHHGHHGGGHHHFHHGGHGGGHHHHHHHHGGGHGDEG<:CGEHHHHHAHHHGHHGGGGGGGGGG:DGGGGADEGFGGGGGFFFFFFFFFFFFFFFFFFFFFFFFFDFFFFFFFFFFFFFFFFFFFFFFFFFAFDFFF;DDFBFFF:FFFFFFFFFFBDFFFFF
@MISEQ03:123:000000000-A40KM:1:1101:13534:1945 1:N:0:TCCACAGGAGT
GGGGGGTGCCAGCAGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTCGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGTCGAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAG
+
DDDDDDBCCFFFGGGGGGGGGGGGHHHHHHHGGGGGHHHHGGGGGHHGGGGGGGHHHHHHGGGGHHHGGGGGGGGGGGGFEHGGHHGGGGGGGGGGGGGGFFFGGGGGGGGFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF:EFFEFFFFFFFFFFFFFFFFFFFFFFFFFFFDDFFFFFFFFFFFFEFFFFFFFFFFFFFAFDFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
@MISEQ03:123:000000000-A40KM:1:1101:14333:2023 1:N:0:TCCACAGGAGT
TTGACGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAG
+
BCCCCCCBCCFFGGGGGGGGGGGGHHHHHHHGGGGGHHHHGGGGGHHGGGGGHGHHHHHHGGGGHHHGGGGGGGGGGGGGGHGHHHHHHHHGHHHHHHHHHHHHGGGGGGHHHHHHHHGFHGHHHHHHHHHGHHHHHHHHHHHHHHHHHGHHHHHHHHGEGGGGGGGFGGFGGGGGGGGGGAGFFFFFFFFFFFEFFFFFFFFFFFFFFFFFFFFFFFFAFAEFFFFFFFFFFFFFFFFB;BFFFFFFFFF
@MISEQ03:123:000000000-A40KM:1:1101:17225:2042 1:N:0:TCCACAGGAGT
TCCCTGTGCCAGCCGCCGCGGTAATACAGAGGTCTCAAGCGTTGTTCGGATTCATTGGGCGTAAAGGGTGCGTAGGTGGCGTGGTAAGTCGGGTGTGAAATTTCGGAGCTTAACTCCGAAACTGCATTCGATACTGCCGTGCTTGAGGACTGGAGAGGAGACTGGAATTTACGGTGTAGCAGTGAAATGCGTAGAGATCGTAAGGAAGACCAGTGGCGAAGGCGGGTCTCTGGACAGTTCCTGACACTGAG
+
CCCCCFFFFFFFGGGGGGGGGGGGHHHHHHHGHHHHHHHHGGGGGHHGGGGGHHHHHHHHGGGGHGHGHGGGFGGGGGGHGGGGHHHHHHGGGHGGGHGHGHHFHGFGGHHHHHHHHGGGGGHHHHHHDHHHHGHHGHGGFGGHHFHHHHHHHHHFGGGGGCGGGFGGGGGGEGCEGGGG0FGGGGGGFFGGFF?9:;;EFFFFFBBFFFFFFFFFFFFAFFFFFFFFFFFFFBABEFFFFFFB;:BFFFF
@MISEQ03:123:000000000-A40KM:1:1101:16728:2094 1:N:0:TCCACAGGAGT
GGGGGGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAG
+
CCCCCCCCCFFFGGGGGGGGGGGGHHHHHHHGGGGGHHHHGGGGGHHGGGGGHHHHHHHHGGGGHHHGGGGGGGGGGGGGFGGGGGGGGGGGGGGGGGGGGFFGGGGGFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFDFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFDFDEBFFFFFFFFFFFFFFFFFFFFFFFAFFFAFFFAEEFFFEFFFFFBBBFFFFFFFFF
@MISEQ03:123:000000000-A40KM:1:1101:15092:2113 1:N:0:TCCACAGGAGT
GGGGCGTGCCAGCAGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAG
+
CCDCDDDDDDFFGGGGGGGGGGGGHHHHHHHGGGGGHHHHGGGGGHHGGGGGHGHHHHHHGGGGHHHGGGGGGGGGGHGFFHGHHHHHHHHGHHHHHHHHHHHHGGGGGGHHHHHHHHGGHGGHHHHHHHHHHHHHHHHFHHHHHHHHHHHHGHGGGGGEEGEGGGGGGGGGGGGGHGHGGFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF@FDFFFFFFFFFFFFFFFFFFFFFFFFFFFFF
@MISEQ03:123:000000000-A40KM:1:1101:15596:2117 1:N:0:TCCACAGGAGT
GTAGGGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAG
+
AABCCCFABFFFGGGGGGGGGGFGHHHHHHHGGGGGHHHHGGGGGHHGGGGGHHHHHHHHGGGGHHHGGGGGGGGGGGGGGHGHHHHHHHHHHHHHHHHHHHHHGGGGGGHHHHHHHHGHHGHHHGHHHHHHHHHHHHHHGHHHHHHHHHHHHGHHHHGEFGGGGGGGGGGGGGGGGGGGGGGGGGFFFFFFFDEFFFFFFFFFFFFFFFFFFFFFFFFADDFFFFFFFFFFFFFFFFFB;BFFFFFFFFF
"""

reads2_string = """@MISEQ03:123:000000000-A40KM:1:1101:14149:1572 2:N:0:TCCACAGGAGT
ACGGACTACTAGGGTTTCTAATCCCGTTTGCTCCCCTGGCTTTCGCTCACTCAGCGTCAGAATTGGTCCAGGCTGCCGCCTTCGCCACCGGTGTTCCTCCAGATATCTACGCATTTCACCGCTACACCTGGAATTCCGCATCCCTCTCCCAACCTCAAGCCATCCAGTATCAGACGCAGTTCCCCGGTTAAGCCGAGGGATTTCACGTCTGACTTAGTTGGCCGCCTACGAGCCCTTTACGCCCAGTAATT
+
BBAABDAFFFFFGGGGGFGGGGHHHGHHHEHHHHGGHFHEGEGHGGGHGGHHHHGFGGGGHGHHHHHHHHHG1GHGHGGGGGHGGGGGHGGGGEEGHHHHFFGGGGHGHFGGGGGHHGHHGGGGGHGFGFHFHGGHH0CDFGHHGHHGHHFHGGGG1FGHGHHHHGEHHGFEHHHCGG?ACHHHH?BGGA0CFGG?BG@GFFB;FF.9AFE/BFFFFBFB/9B-;=.AF?.A<BAB/9BF.ABBF.FBBFF
@MISEQ03:123:000000000-A40KM:1:1101:14170:1596 2:N:0:TCCACAGGAGT
ACGGACTACACGGGTTTCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGCGTCAATATGTGTCCAGTGCGTCGCCTTCGCCACTGGTGTTCTTCCGAATATCTACGAATTTCACCTCTACACTCGGAATTCCACTCACCTCTCCACAATTCTAGCGATGCAGTCTTAAAGGCAGTTCCGGAGTTGAGCCCCGGGCTTTCACCTCTAACTTACAAAGCCGCCTACGTGCGCTTTACGCCCAGTAATTC
+
BBBBBBBFFFFBGGGGGGGGGGHHHHHHHHGHHHGGHGGGGGGHGGGGGHHHGHGGGGGHHFHHHHHHHHHH5EFGGGGGGHHGGGGHFHFHHGHHHHHGGGGGHHHFHGGEHHHHFHHHHHHHHHHHGGGGGHHHHHHHHHHHHHHHHHHHHHHHHHHGGHGGHHHGHHHHGHHHHGHHHHGGECGHFHFGGGFGFGGGGGGGFGFFGFGFFFFFFFFBFF->=AFF.F.A=FFFFFFEDFFA.B/BBB#
@MISEQ03:123:000000000-A40KM:1:1101:14740:1607 2:N:0:TCCACAGGAGT
ACGGACTACTGGGGTTTCTAATCCTGTTTGCTCCCCACGCTTTCGTGCATGAGCGTCAATATTATCCCAGGGGGTTGCCTTCGCCATCGGTGTTCCTCCGCATCTCTACGCATTTCACTGCTACACGCGGAATTCCACCCCCCTCTGCCACACTCAAGCTCGCCAGTTTCCACTGCCATTCCCAGGTTGAGCCCGGGGATTTCACAGCAGACACAACGAAACGCCTGCGCACGCTTTACGCCCAGTAATTC
+
AABBBBDFFFFFGGFFGGBGFGHHFGHFHCHCHHGGHGFGGGGHGHEFHHHHHHFGGGGFHFFF5FGFHHGGGCEEEHHGHHHGGGGHGDFGGEHHHHEEGGGGHHHHHGGGGGHFGHHFHHHHHHD/C/>@FHHHHHGGGCGGGHHHGHGFGHHGHHFHGF-@E-BFGGFFBFGGGGGGFGGB0.;C/BFFFFF;-=CDFFFFFFFFF?FFFFFF?@EF.@9@DFA.;D@---9.AFF??=DF?9/9FF#
@MISEQ03:123:000000000-A40KM:1:1101:14875:1613 2:N:0:TCCACAGGAGT
ACGGACTACAGGGGTATCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTTGTCAGTTTTGAATGCAGTTCCCAGGTTGAGCCCGGGGATTTCACATCCAACTTAACAAACCACCTACGCGCGCTTTACGCCCAGTAATTC
+
BBBBBBBFFFFBGGGGGGGGGGHHHHHHHHHHHHGGHGGGGGGHGGGGGHHHHHHHHHHHHHHHHHHHHHHHHGGHGGGGGHHGFGGHBGHHHGHHHHHHHHHHHHHHHGGGGGHHHGHGGGGGHHHHHHHHHGHHHHHHHGGHHHHHHGFGGDHHEFHHFHGGHHHHHHGGHHEHHHHHHHFHHGHHHGBGHGGFCGGGGGGGGGGGGBFGGGGGGGG..CAGGFFBA;@=9@FFFFF?.BCFF/9FFF#
@MISEQ03:123:000000000-A40KM:1:1101:14902:1627 2:N:0:TCCACAGGAGT
ACGGACTACCGGGGTTTCTAATCCTGTTTGCTCCCCTAGCTTTCGCGCCTCAGCGTCAGTTGCGGTCCAGTGCTTCGCCTTCGCCTCTGGTGTTCCTCCAGATATCTACGCATTTCACCGCTACACCTGGAGTTCCAATCACCCCTCCCGCACTCAAGCTCGGCAGTTTCGGAGGCACACCGCAGTTGAGCTGCAGGCTTTCACCCCCGAATTGCCAAGCCGCCTACGCGCGCTTTATGCCCAGTAAATCC
+
BBBBBDDFFFBBGGAEEGGGGGHHHHHHHHHGHHGGHGGGHHHHFGEFGGGHFHGGGGGHHGBEEGGGHHHHEHHH?EEGGHHGGGGHFHHHHFHHHHHHFHHHHHHHHGGGGGHGHHHGGGGGHHHHGEFFHHHHHBHHHHGHGGGGG?CGGGHGHGHHHGDGCD:;BFDGG?ADGGGGG-;-9:FFEFFFBF/BFFFFFF/FF.9;-9-.;/;BFB.:E--@AAF.--;-9B9:9B;9BFBAFFFB/9B
@MISEQ03:123:000000000-A40KM:1:1101:15883:1644 2:N:0:TCCACAGGAGT
ACGGACTACAAGGGTTTCTAATCCTGTTTGCTCCCCTGGCTTTCGTGCCTCAGTGTCAGGTCTTGTCCAGGGACTCGCCTTCGCCACTGGTGTTCCACTTGATATCTACGCATTTCACTGCTACACCAAGTATTCCAGTCCCCCCTCCAAGCCTCCAGCGGAATAGTATCCCGTGACTTTTCGCAGTTAAGCTGCGAGCTTTCACACGAGAATTACCCCACCACCTACGCACCCTTTACGCCCAGTGAATC
+
BBBBBBBFFFFFGGGGGGGGGGHHHHGHHHHHHHGGHGGGHHHHGHEFHHHHHHHHHHHHHHHHHHHHFGHG1FGHGGGGGHHGGGGHHGFGCEHGHHHHHHHHHHHHHGGGGGHHHHHHHHGHGHHHHGGHHHHHGFHGHHFFGCGGGHGFGHHH0GFGFCCFHH>DGGHHFCDEGHHGHH-C:;ECCHCBH0;@@-AACFFFGB/DC-C.;B00FGGFDF.AEFFF.BADAF?FFBF############
@MISEQ03:123:000000000-A40KM:1:1101:17346:1668 2:N:0:TCCACAGGAGT
ACGGACTACCCGGGTATCTAATCCTGTTTGCTCCCCACGCTTTCGCGTCTCAGCGTCAGTTACCGTCCAGAAAGCCGCCTTCGCCACTGGTGTTCCTCCTGATATCTACGAATTTCACCTCTACACCAGGAATTCCGCTTTCCTCTCCGGCACTCAAGAAAGGAAGTTTCGAGTGCAGTTCCTGGGTTGAGCCCAGGGATTTCACGCCCGACTTTCCTTCTCGCCTACACGCCCTTTACGCCCAGTGTTTC
+
BB@AADDFFFCAGGGGGGFFGG4DFGHHFHHHHHEEHGGGGGGHGGGGGGHHHH?AEFGHFHDFGG1??FGHFFFHCEGGGHA/EGFHGHHHGEFHHFHFFHHEG4GGHGGFGEFGHHHFF1DGFFGHGGAGFFGFFDEAF?FGF<0<<DGCDHHFFHFF/GF/B.CCGF?CGCCGGFC;CGBHH.CFGBBBE..CEB;;FBBBBE?9?@?9-99FFBB9/;.E-9.9;BD-.9>B999..;@FFB#####
@MISEQ03:123:000000000-A40KM:1:1101:14457:1707 2:N:0:TCCACAGGAGT
ACGGACTACAGGGGTATCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCTCTGGTGTTCCTCCCGATCTCTACGCATTTCACCGCTACACCGGGAATTCCACGACCCTCCCCAGAACTCTAGTCTCACCGTTTCTGGGGCACTTCTCGAGTTAAGCTCGAGGCTTTCACCCCAGACAAATGAGACCACCTACCTACCCTTTACGCCCAATAATTC
+
BCCCBBBFFFFFGGGGGGGGGGHHHHHHHHHHHHGGHGGGGGGHGGGGGHHHHHHHHHHHHFHHHHHHHHHHHGGHGGGGGHGGGGGHHHHHHGHHHHHGGGGGHHHHHGGGGGHHHHHGGGGGHHHGGG@GHHHHGHDGGGGHHGGGHAEHHHHHHGHGHHGHHDCEHHHHGHGGGGGGGFG?C.FEFFBFFFEDG?GGGGFFFFFFF?EFF..BFBBFFFFFFAFBFFFFFFFFFBF.9D@FEFEFB;9
@MISEQ03:123:000000000-A40KM:1:1101:15133:1720 2:N:0:TCCACAGGAGT
ACGGACTACAAGGGTATCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTTGTCAGTTTTGAATGCAGTTCCCAGGTTGAGCCCGGGGATTTAACATCCAACTTAACAAACCACCTACGCGCGCTTTACGCCCAGTAATTC
+
BBBBBBBFFFFFGGGGGGGGGGHHHHHHHHHHHHGGHGGGGGGHGGGGGHHHHHHHHHHHHHHHHHHHHHHHHGGHGGGGGHHGGGGHHHHGHGHHHHHHHHHHHHHHHGGGGGHHHHHGGGGGFHHHHGHHHHGHHHHHHGGHHHHHHHHHHHHHHHHHHHHHHHGHHHEGHHHHHHHHHHDGHFHGFHHHHGGGGGGGGG000;FFBFGGBBF9FGGGGE.9:.BFADBA;DDFFFFEDBF?EFFFBFB
@MISEQ03:123:000000000-A40KM:1:1101:16328:1722 2:N:0:TCCACAGGAGT
ACGGACTACAGGGGTATCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTTGTCAGTTTTGAATGCAGTTCCCAGGTTGAGCCCGGGGCTTTCACATCCAACTTAACAAACCACCTACGCGCGCTTTACGCCCAGTAATTC
+
ABBABBBFFFFFGGGGGGGGGGHHHHHHHHHHHHGGHGCEGGGHFGGGGHHHHHHHHHGFHHHHHHHFHFEFHGGHGGGGGHHFFEGHHFGFHEHHHGHHHHHHHGHHHFGGGGHHHHHGGGGGHHHHHHGHHHHHHHHHHGGGHHHHHHHGHHHHHBHHHHHHGGHFHFFGHHHFHHHHGHFGHEGGGHFHHGGGC?A?AGFGFGGGGGGGGGGGG09CG9AEFGFF.@-@->DFFFFAFFCFEF/BFBF
@MISEQ03:123:000000000-A40KM:1:1101:17115:1764 2:N:0:TCCACAGGAGT
ACGGACTACAGGGGTATCTAATCCTGTTCGCTACCCACGCTTTCGTCCATCAGCGTCAATCCATTAGTAGTAACCTGCCTTCGCAATTGGTATTCCATGTAATCTCTAAGCATTTCACCGCTACACTACATATTCTAGTTACTTCCTAATAATTCAAGTTTAACAGTATCAATGGCCGTTCCACCGTTGAGCGATGGGCTTTCACCACTGACTTATTAAACCGCCTACGGACCCTTTAAACCCAATGATTC
+
ABBCCCBFFFFFGGGGGGGGGGHHHHHHHGGGGGHHHGGEGGGHGHGGGHHHHHGGGGGHHHHHHHHHHHHHHHHHHHHHHHHGGGGHHHHHHHHGHHFHHHHHHHHHGHHHHHHFGHHGGGGGHHHHHHHHHHHFHHHHGHHHHGHFFHHHHHHHHHHHHGHHGHFGFFHHHHHHHGGGHHHHHGHGGFHHGGHGGEFHHHHHHGCHHBGGGFHHHHGGGFCGGFD=EA.-BDFGG0BBFF.EFFBFFF:
@MISEQ03:123:000000000-A40KM:1:1101:14429:1791 2:N:0:TCCACAGGAGT
ACGGACTACAGGGGTATCTAATCCTGTTTGCTCCCCACGCTTTCGTGCATGAGCGTCAGTACAGGTCCAGGGGATTGCCTTCGCCATCGGTGTTCCTCCGCATATCTACGCATTTCACTGCTACACGCGGAATTCCATCCCCCTCTACCGTACTCTAGCTATGCAGTCACAGATGCAGTTCCCAGGTTGAGCCCGGGGATTTCACAACTGTCTTACATAACCGCCTGCGCACGCTTTACGCCCAGTAATTC
+
ABBBBBBFFFFFGGGGGGGGGGHHHHHHHHHHHHGGHGGGGGGHGHCFHHHHGHCGGGGHHDHHHFFHHHHGGGGHHHHHHHEGGGGHGDAEGGHHHHHDGGGGHHHHHGGGGGHHHHHHHHHHHHGGGGGGGHFHHHHHHGGGGGHHHCFHGHHHHHHGHHHHHHCHHHHGHHFHHFGFFGGCGAFGBBBBFGGGGGGDFFGGGFFFF/BFBBFFBFFFFF>>DDFFBFFFFFFFFFE..@DDA;9FFF#
@MISEQ03:123:000000000-A40KM:1:1101:14969:1845 2:N:0:TCCACAGGAGT
ACGGACTACCGGGGTATCTAATCCTGTTCGCTCCCTACGCTTTCGCGTCTCAGCGTCAGTACCGTCCCAGAGCGCTGCTTTCGCCATAGGTGTTCTTCCTGATATCTGCGCATTTCACCGCTACACCAGGAATTCCACTCTCCTCTTCCGGACTCTAGCCAGATGGTTTCCGATGACATCTCGAGGTTAAGCCTCGAGTTTTCACACCGGACCTACCTGGCCGCCTACACGCTCTTTACGCCCAATAAATC
+
AA?33ADFFBDDGGEGGGGGCGHHHHGHFGGGHDFHGHGGDGGHGGGGGGGHEEAGGGFHHHHGCEEFHEGHEFGGGGHHHHGGGGGH3BFEGEHHHHHHF3GGHGHFGGGGGGHBGHHGGGGGHHHGHGHHHHFHHHHHHGGGHHHHHEG?CFHHHFHCFGCBGF;GHHH?DDGFGBCBFGFBAAGFFFFGGFGFA.C?FBBB0;BAC-@--.9BFFFFF.-9@.DFBFDF###################
@MISEQ03:123:000000000-A40KM:1:1101:17499:1855 2:N:0:TCCACAGGAGT
ACGGACTACCAGGGTTTCTAATCCTGTTCGCTACCCACGCTTTCGTCCATCAGCGTCAATCCATTAGTAGTAACCTGCCTTCGCAATTGGTATTCCATGTAATCTCTAAGCATTTCACCGCTACACTACATATTCTAGTTACTTCCTAATAATTCAAGTCTAGCAGTATCAATGGCCGTTCCACCGTTGAGCGATGGGCTTTCACCACTGACTTACTAAACCGCCTACGGACCCTTTAAACCCAATGATTC
+
ABAAADBFFFFFGGGGEFGGGGHHHFHHHGGGGGHHHGGGGGGHGHGGGHHGHGGGGGGHHHHHHFHHHHGHHHHHHFHHHHHGGGFHGFGHHHHHHHFGGHHHHHHHBFGHHHHHHHHGGFGGHHHHHHHHHGHHHFHHFHHHHDGHHHHGHHHGHHFHHHHHFHFFDGHHHHFHFCCGHBFGC.GED./CFCH..C.CHHHHHGHHEGGHHHHHHHHGGGGGAGGGGA?@DGFFBFGFFFFEABF/BB#
@MISEQ03:123:000000000-A40KM:1:1101:14009:1889 2:N:0:TCCACAGGAGT
ACGGACTACCCGGGTACTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTTGTCAGTTTTGAATGCAGTTCCCAGGTTGAGCCCGGGGCTTTCACATCCAACTTAACAAACCACCTACGCGCGCTTTACGCCCAGTAATTCC
+
BBBBBBBFFFFBGGGGGGGGGGHHHHHHFHGHHGGFGGGGGGHEGGGGGHHHHHFHHHHGFHHHHGHFHHGHGGHFGGGGHGGGGGHHHFGHGHHHHHHGHHHHHHHHGGGGGHHHFHAFGGGHGHGHHHHHHHHHHHHHGGFHHHHGHHHHHHGHHHHHGHFHHGGHHHGHHHHHHHGDGHHHGHHHFHHHEGGGFGGEGB0;FCFGGGBBFFFFGGGGEGFEF//9@;BFF9AABF?ADFFFFFFBB/B
@MISEQ03:123:000000000-A40KM:1:1101:17869:1908 2:N:0:TCCACAGGAGT
ACGGACTACCGGGGTTTCTAATCCCGTTTGCTCCCCTGGCTTTCGTGCCTCAGCGTCAGGAAATGTCCAGAGACTCGCCTTCGCCACTGGTGTTCCTCTCGATATCTACGCATTTCACTGCTACACCGAGAATTCCAGTCTCCCCTCCATTCCTCTAGCCGAATAGTATCCGACGCAGTTTCCGAGTTAGGCTCGGAGATTTCACGTCAGACTTACCCGACCGCCTACGCACCCTTTACGCCCAGTGAATC
+
BBBBBBBFFFBBGGGGFGGGGGHHHGHHHGHHHHGGHGHHHHHHGFGGHHHHHHGGGGGHHHHHHHHHHHGHHHHHGCGGGHHGGGGHHHGHHGHGHHHHGHFHEHHHHGGGGGHGFHHGGHHHHHHGGGGGHHFGFBGHFFGHGHDDHHFHDGHEHHGHGCCEEG<D0DGGGGGGGGDGGGGGEBEAFFF/CF?-.--BFFBFFFFFEDFFFFFFFF?;->AB;.9B.;.9@AEBF/:9AC-DFFFB###
@MISEQ03:123:000000000-A40KM:1:1101:14767:1933 2:N:0:TCCACAGGAGT
ACGGACTACCGGGGTTTCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGCGTCAACAGTCGTCCAGTGCGCCGCCTTCGCCACTGGTGTTCTTCCGAATATCTACGAATTTCACCTCTACACGCGGAATTCCACCCCCCTCTCCCACGTTCAAGCTATGCAGTCTTACAGGCAGTTCCCAGGTTGAGCCCGGGGATTTCACAACTCTCTTACGAAACCGCCTGCGCACGCTTCACCCCCAGTAATTC
+
1>>AAD1C>CDDG0EAEGBBGGDDGDGDF0FFHHCFGG/EEECGG/E?/A0BG1AE//>1@>FBEFE>/G@EF@?EEEEEEGFEA//F1<G0FEF2BFF//?F/BDD2?<//?FFHDDGH0DGHDG//<-><CG0=DG.<--;C.:CC/;.9.;9/0C0/0;99B0F/00C00999C.:0F0C90-;FF/99B-9@---;BBFEFF#############################################
@MISEQ03:123:000000000-A40KM:1:1101:17050:1939 2:N:0:TCCACAGGAGT
ACGGACTACAGGGGTTTCTAATCCCGTTTGCTCCCCTGGCTTTCGCGCCTCAGCGTCAGTATCAGCCCAGAGACTCGCCTTCGCCACTGGTGTTCCTCCCGATATCTATGAATTTCACCGCTACACCGTCAATTCCGCCGACTTCATCTGTACTCAAGCCCCGCAGTATCAATGCCACCCTCCCGGTTAAGCCGGGATATTTCAGCACTGACTTACAGGGCCGCCTACGCACCCTTTAAACCCAATAAATC
+
BBBBBBBFFFFFGGEEGGGGGGHHHGHHHGHHHHGGHGHHHHHHGGGGGGGHHHGGGGGHHHHHHHHGHGHHHHHHGGGGGHGGGGGHHFHHEFHHHHHGGGGGGHHHHHHHHHHHHHHGGGGGHHHGHGGGHHHHHGGGGGGGHHHGHGHGHHHHHHGGHGGGGGGGFHGGGGGGGGGGGGGGGAFGFFFFGAAFFFFFFFFFFFF.BFFFFFFFFFFFFFBFFF.B9DFFFD.FF/B9/BDFEFFF###
@MISEQ03:123:000000000-A40KM:1:1101:16782:1940 2:N:0:TCCACAGGAGT
ACGGACTACAAGGGTATCTAATCCTGTTTGCTCCCCACGCTTTCGCGTCTCAGCGTCAGTCACGTACCAGAGAGCCGCCTTCGCCACGGGTGTTCTTCCTGATATCTGCGCATTTCACCGCTACACCAGGAATTCCACTCTCCTCTTCCGGACTCTAGCTAGATGGTTTCCATCGACGTCTCGAGGTTGAGCCTCGAGTTTTCACAACGGACCTACCTAGCCGCCTACACGCTCTTTACGCCCAATAAATC
+
BBBBBBBFFFFFGGGGGGGGGGHHHHHHHHHHHHGGHGFGGGGHGGGGGGHHHHGGGGGHHHHGHGHGHHHEHGHHGGGGGHHGGGGGGGGGGFFHHHHHHHHHHHGHHGGGGGHHHHHGGGGGHHHGHGEHFHGHHHHHGHHHHHHFHGGGGGHHHGHHBCGHHGHHGHHHHGGGGGGGGGGGFAGGGFFGGFGGDDBCFGFFBF/DFFFFFFFFF99:BFDFFFDFFF.@DFEFFFF?FDFDEF/B###
@MISEQ03:123:000000000-A40KM:1:1101:13534:1945 2:N:0:TCCACAGGAGT
ACGGACTACTGGGGTTTCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTCGACAGTTTTGAATGCAGTTCCCAGGTTGAGCCCGGGGATTTCACATCCAACTTAACGAACCACCTACGCGCGCTTTACGCCCAGTAATTC
+
BBBBBBBFFFFFGGEEGGGGGGHHHHHHHHHHHHGGHGGGGGGHGGGGGHHHHHHHHHHHHGHHHHHGGHHGHGGHGGGGGHHGGGGHHGGGHFHHHHHHHHHHHHHHHGGGGGHHHHHGGGGGHHHGHHHHHHHHHHHHHGGHHHHHGHHHHHHHHHHHHGGGGGHHFHHGHHBGHHHHGHHHHGHHGFFHHGGGGGGGFGBFGGGGFFFGGGFFFGGGFBFFFAFFDFFFFACFFFFEFFFDF/BFFB;
@MISEQ03:123:000000000-A40KM:1:1101:14333:2023 2:N:0:TCCACAGGAGT
ACGGACTACCGGGGTATCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTTGTCAGTTTTGAATGCAGTTCCCAGGTTGAGCCCGGGGCTTTCACATCCAACTTAACAAACCACCTACGCGCGCTTTACGCCCAGTAATTC
+
BCBBCBCFFFCCGGGGGGGGGGHHHHHHHHHHHHGGHGCGGGGHGGGGGHHHHHHHHHHHHHHHHHHHHHHHHGGHGGGGGHHGGGGHHHHHHEHHHHHHHHHHHHHHHFGGGGHHFHHGGGGGHHHHHHHHHHHHHHHHHGGHHHHHHHGHHHHHHHHFHHHHHHHHHHFGHHHHHGHHHHHHHGHHFFFHHGGGGGGGGGFGFGGGGGGGFGGGBFFFFE.AEA/B:DFA-;:DF;/AFC-..B/BFF#
@MISEQ03:123:000000000-A40KM:1:1101:17225:2042 2:N:0:TCCACAGGAGT
ACGGACTACCAGGGTTTCTAATCCCGTTTGCTCCCCTGGCCTTCGTGCCTCAGTGTCAGGAACTGTCCAGAGACCCGCCTTCGCCACTGGTCTTCCTTACGATCTCTACGCATTTCACTGCTACACCGTAAATTCCAGTCTCCTCTCCAGTCCTCAAGCACGGCAGTATCGAATGCAGTTTCGGAGTTAAGCTCCGAAATTTCACACCCGACTTACCACGCCACCTACGCACCCTTTACGCCCAATGAATC
+
BBBBBBBFFFFFGGGGGGGGGGHHHGHHHGHHHHGGHEHHHHHHHHGGHHHHHHHHHHHGHHHHHHHHHHHHGHGHGGGGGHHGGGGHHHHHHHHHHHHHGGHHHHGFHGGGGGHHHHHHGHGHHHHGHHDEGHHHHBHGHHHHHHHHHHFFGHHHHHFHH?@CGCGHGGGHHHHHFHHHHHGGDCGGHHGGGHHGGGCHHGFEFGGDGGGGCDFGBBBEE9.;99BFDFDFFFFFFFFEDFDDEFFFF/B
@MISEQ03:123:000000000-A40KM:1:1101:16728:2094 2:N:0:TCCACAGGAGT
ACGGACTACACGGGTTTCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTTGTCAGTTTTGAATGCAGTTCCCAGGTTGAGCCCGGGGATTTCACATCCAACTTAACAAACCACCTACGCGCGCTTTACGCCCAGTAATTC
+
ABBCCCCFFFFCGGGGGGGGGGHHHGHHHHHHHGGGHGGGGGGHGGGGGHHHHHHHHHHHHHHHHHHGHHGHHGGHGGGGGHHGGGGHHHHHHGHHHHHHHGHHHHHHHGGGGGHFFHHGGGGGHHHHHGHHHHHGHHGHHGGHHHGHHHGHHHHHHGHHHHHHHHHHHHHGHHHHHHHHHHHGFEHHGFHHHGGGGGGGGGGFGGGGGGGGGGF0FGGGF?9EEFFFB-;DFFFFFFFADCFFFFFFFFB
@MISEQ03:123:000000000-A40KM:1:1101:15092:2113 2:N:0:TCCACAGGAGT
ACGGACTACTGGGGTATCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTTGTCAGTTTTGAATGCAGTTCCCAGGTTGAGCCCGGGGCTTTCACATCCAACTTAACAAACCACCTACGCGCGCTTTACGCCCAGTAATTC
+
BCCCCBCFFFFFGGGGGGGGGGHHHHHHHHHHHHGGHGGGGGGHGGGGGHHGHHHHHHHHHGHHHHHHHHHGHGGGGGGGGHHGGDGHHHHGHFHHHHHHHHHHHHHHHFGGGGHHFHHGGGGGHHHHHHHHHHHFFHHHHGGHHHHHHHGHHHHHHHHHHHGHHGHHHHHGHHHHHHHHHFGHHGCGCGHGHEGGGFGAEFFFGGGFGFGGGGGGGGGG?EG.CFGFF-@ACFFFAFFADFFFEFFFBF9
@MISEQ03:123:000000000-A40KM:1:1101:15596:2117 2:N:0:TCCACAGGAGT
ACGGACTACCAGGGTATCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGTGTCAGTATCAGTCCAGGTGGTCGCCTTCGCCACTGGTGTTCCTTCCTATATCTACGCATTTCACCGCTACACAGGAAATTCCACCACCCTCTACCATACTCTAGCTTGTCAGTTTTGAATGCAGTTCCCAGGTTGAGCCCGGGGATTTCACATCCAACTTAACAAACCACCTACGCGCGCTTTACGCCCAGTAATTC
+
AAAAABDFFFFFGGGGGGGGGGHHHHHHHHHHHHGGHGCFGGGHGGGGGHHHHHHHHHHHFHHHFDHHHHGHHGGHGGGGGHEGG?FHGHHHHGHHHHHHHGHGHHFHHECFFEHHGHHGG@GGHHHGGHHHHHHHHHFHHGGHHGDHHHHHHHHHHGHHHHHFHFHHFE0<FDDFGHHHHGBFHFHHCFBCGECF?@DDFGGGGGGGGGBFFFB;FFEGG?EGGFFF.@;A9-BFFFF9A@@D?FFFBF9
"""


expected_assembly = """@MISEQ03:123:000000000-A40KM:1:1101:14149:1572 1:N:0:TCCACAGGAGT
GGGGGGTGCCAGCCGCCGCGGTAATACGGGGGGGGCAAGCGTTGTTCGGAATTACTGGGCGTAAAGGGCTCGTAGGCGGCCAACTAAGTCAGACGTGAAATCCCTCGGCTTAACCGGGGAACTGCGTCTGATACTGGATGGCTTGAGGTTGGGAGAGGGATGCGGAATTCCAGGTGTAGCGGTGAAATGCGTAGATATCTGGAGGAACACCGGTGGCGAAGGCGGCATCCTGGACCAATTCTGACGCTGAGTGAGCGAAAGCCAGGGGAGCAAACGGGATTAGAAACCCTAGTAGTCCGT
+
CCCCCCCCCFFFGGGGGGGGGGGGHHHHGGGGGFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF-BFFFFFFFFFF9FFFFFFGFGFFGGFFFFGGFFHHHHFFFGGFHHHFFGHHFGHHHHGHGFFGGGGHFHHGHHGHHGFFFEHHGGHFHFGFGHGGGGGHHGHHGGGGGFHGHGGGGFFHHHHGFFGGGGHGGGGGHGGGGGHGHG1GHHHHHHHHHGHGGGGFGHHHHGGHGGGHGEGEHFHGGHHHHEHHHGHHHGGGGFGGGGGFFFFFADBAABB
@MISEQ03:123:000000000-A40KM:1:1101:14170:1596 1:N:0:TCCACAGGAGT
ATGGCGTGCCAGCAGCCGCGGTAATACGGAGGGGGCTAGCGTTGTTCGGAATTACTGGGCGTAAAGCGCACGTAGGCGGCTTTGTAAGTTAGAGGTGAAAGCCCGGGGCTCAACTCCGGAACTGCCTTTAAGACTGCATCGCTAGAATTGTGGAGAGGTGAGTGGAATTCCGAGTGTAGAGGTGAAATTCGTAGATATTCGGAAGAACACCAGTGGCGAAGGCGACTCACTGGACACATATTGACGCTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCGTGTAGTCCGT
+
CCCCCCCCCCFFGGGGGGGGGGGGGHHHGGGGGGGGGHHHGGGGGHHGGGGGHHHHHHHHGGGGHHHGGGGGHHHGHGGGGGHHHHHHHHGHGHHHHHHGHHHHGGGGGHHHHHHHHHHHHGHHHHHHHHHGHHHGGHGGHHHHHHHHHHHHHHHHHHHHHHHHHHGGGGGHHHHHHHHHHHFHHHHFGGHFHHHGGGGGHHHHHGHHFHFHGGGGHHGGGGGGFE5HHHHHHHHHHFHHGGGGGHGHHHGGGGGHGGGGGGHGGHHHGHHHHHHHHGGGGGGGGGGBFFFFBBBBBBB
@MISEQ03:123:000000000-A40KM:1:1101:14740:1607 1:N:0:TCCACAGGAGT
AGTGTGTGCCAGCAGCCGCGGTAATACGTAGGGTGCGAGCGTTAATCGGAATTACTGGGCGTAAAGCGTGCGCAGGCGGTTCGTTGTGTCTGCTGTGAAATCCCCGGGCTCAACCTGGGAATGGCAGTGGAAACTGGCGAGCTTGAGTGTGGCAGAGGGGGGTGGAATTCCGCGTGTAGCAGTGAAATGCGTAGAGATGCGGAGGAACACCGATGGCGAAGGCAACCCCCTGGGATAATATTGACGCTCATGCACGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCAGTAGTCCGT
+
AABCCFFFFFFFGGGGGGGGGGGGHHHHHHHEGGFG2EEGGGGGGHHGGGGGHGHHHHHHGGGGHHHGGGGGGGGGGGGFGGGGHFGFGBGGFHFGFFHHGHHHGGGGFCHHHHHFGGGGGGGHGHGGGGGHH/EDHFGHGHHHHHGFGHGHHHGGGEGGGHHHHHFFFFF/DHHHHHHFHHGFHGGGGGHHHHHGGGGEEHHHHEGGFDGHGGGGHHHGHHEEEFGGGHHFGF5FFFHFGGGGFHHHHHHFEHGHGGGGFGHGGHHCHCHFHGFHHGFGBGGFFGGFFFFFDBBBBAA
@MISEQ03:123:000000000-A40KM:1:1101:14875:1613 1:N:0:TCCACAGGAGT
GGTGGGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCTGTAGTCCGT
+
CCCCCCCCCFFFGGGGGGGGGGGGGHHHHHHGGGGGHHHHGGGGGHHGGGGGHHHHHHHHGGGGHHHGGGGGGGGGHHGGGHGHHHHHHHHHHHHHHHHHGHHHGGHGGGHHHHHHHHHHHHHHHHHHHHHHHHHGHHHHHHHHHHHHHHHHHHHHHHHHHHHHHGHHHHHHHHHGGGGGHGHHHGGGGGHHHHHHHHHHHHHHHGHHHGFHGGFGHHGGGGGHGGHHHHHHHHHHHHHHHHHHHHHHHHGGGGGHGGGGGGHGGHHHHHHHHHHHHGGGGGGGGGGBFFFFBBBBBBB
@MISEQ03:123:000000000-A40KM:1:1101:14902:1627 1:N:0:TCCACAGGAGT
AGGGGGTGCCAGCCGCCGCGGTAAGACGAAGGTGGCTAGCGTTATTCGGATTTACTGGGCATAAAGCGCGCGTAGGCGGCTTGGCAAGTCGGAGGTGAAAGCCTGCAGCTCAACTGCGGTGTGCCTCCGAAACTGCCGAGCTTGAGTGCGGGAGGGGTGATTGGAACTCCAGGTGTAGCGGTGAAATGCGTAGATATCTGGAGGAACACCAGAGGCGAAGGCGAATCACTGGACCGCAACTGACGCTGAGGCGCGAAAGCTAGGGGAGCAAACAGGATTAGAAACCCCGGTAGTCCGT
+
ABBBBBBBBFFCFGGGGGGGGGGGHHHGGGGGHHGHHHHHGGGGHHHHHGEGHHHHHHHHHHHHHHHGGGGGGGGGHGGGGGGGHFHGHHGGGGGGHHHHHHFHHHHHHHHHHHHCGGGGGGGHHHHGGGDGDGGGGHHHGHGHGGGFFGGGGGHGHHHHFHHHHHFFFGHHHHGGGGGHHHGHGGGGGHHHHHHHHFHHHHHHFHHHHFHGGGGHHGGFFFHHHEHHHHGGGFFFGHHGGGGGHFHGGGFEGFHHHHGGGHGGHHGHHHHHHHHHGGGGGEEAGGBBFFFDDBBBBB
@MISEQ03:123:000000000-A40KM:1:1101:15883:1644 1:N:0:TCCACAGGAGT
GGGATGTGCCAGCAGCCGCGGTAATACAGAGGTCCCGAGTGTTGTTCGGATTCACTGGGCGTAAAGGGTGCGTAGGTGGTGGGGTAAGTCTCGTGTGAAAGCTCGCAGCTTAACTGCGAAAAGTCACGGGATACTATTCCGCTGGAGGCTTGGAGGGGGGACTGGAATACTTGGTGTAGCAGTGAAATGCGTAGATATCAAGTGGAACACCAGTGGCGAAGGCGAGTCCCTGGACAAGACCTGACACTGAGGCACGAAAGCCAGGGGAGCAAACAGGATTAGAAACCCTTGTAGTCCGT
+
CDDDDDFFFFFFGGGGGGGGGGGGHHHHHHHGHGGH2EFGHHHHHHHGGGGGHHHHHHHHGGGGHHHGHGGGGGGGHHGGGGGGGGHHHHHHHGHHHHHHHHHGGGHGGHHHHHHDGHHGHHHHHHHHHGGGHHHHHHHHGGGHHHHGGHHGGGHGGHHGHFGHHHHHGGHHHHGHGHHHHHHHHGGGGGHHHHHHHHHHHHHGHFFGFGHHGGGGHHGGGGGHGFDGHGFHHHHHHHHHHHHHHHHHHHHFEHGHHHHGGGHGGHHHHHHHGHHHHGGGGGGGGGGFFFFFBBBBBBB
@MISEQ03:123:000000000-A40KM:1:1101:17346:1668 1:N:0:TCCACAGGAGT
GGCGGGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAACGTTGCTCGGAATCACTGGGCGTAAAGGGCGTGTAGGCGGGAAGGAAAGTCGGGCGTGAAATCCCTGGGCTCAACCCAGGAACTGCACTCGAAACTTCCTTTCTTGAGTGCCGGAGAGGAAAGCGGAATTCCTGGTGTAGAGGTGAAATTCGTAGATATCAGGAGGAACACCAGTGGCGAAGGCGGCTTTCTGGACGGTAACTGACGCTGAGACGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCGGGTAGTCCGT
+
BCCCCDDBCFFFGGGGGGGGGGGGHHHHHGGGGAFEHFHHGHHHCFHGGEGGHHHGHGHHGGGGHHHGEEEGEFHFFGGGBBFF?GGF?FDGFGGCGDGHFHHHHGHGGHGGHHHHBGE;CFGGFCGFFFGFG?CFFGFGGHFFHHDCGFFFA<FGFFFFFDFFGFFGFGGHGFFGD1FFHHHGFFGFGGHGGEGEHHFFHFHHFEGHHHGHFGFFFHGGGFEHFFFHGF?BBGGFDHFHGFEA?HHHHGGGGGGHGGGGGGHEEHHHHHFHHGFD4GGFFGGGGGGACFFFDDAA@BB
@MISEQ03:123:000000000-A40KM:1:1101:14457:1707 1:N:0:TCCACAGGAGT
TGGGTGTGCCAGCCGCCGCGGTAATACGAGGGGTGCAAGCGTTGCTCGGAATTATTGGGCGTAAAGGGTAGGTAGGTGGTCTCATTTGTCTGGGGTGAAAGCCTCGAGCTTAACTCGAGAAGTGCCCCAGAAACGGTGAGACTAGAGTTCTGGGGAGGGTCGTGGAATTCCCGGTGTAGCGGTGAAATGCGTAGAGATCGGGAGGAACACCAGAGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCTGTAGTCCGT
+
AABCBCCCFFFFGGGGGGGGGGGGHHHHGGGGGGGGHHHHGGGGGHHGGFGGHHHHHHHHGGGGHHHGHHGGHHHGHHGGHHFBFHHHHHHHHHGGGHHHHHHHHGGGGGHHHHHHGHGGGHHHHHHHHHHHHHHHGHHHHHHHHHHHHHHHHHHHGGGGGHGHHHHGGGGGHHHGGGGGHHHHHGGGGGHHHHHGGGGGHHHHHGHHHHHHGGGGGHGGGGGHGGHHHHHHHHHHHFHHHHHHHHHHHHGGGGGHGGGGGGHGGHHHHHHHHHHHHGGGGGGGGGGFFFFFBBBCCCB
@MISEQ03:123:000000000-A40KM:1:1101:15133:1720 1:N:0:TCCACAGGAGT
TATAGGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCTTGTAGTCCGT
+
DDDDDFFFFFFFGGGGGGGGGGGGHHHHHHHGGGGGHHHHGGGGGHHGGGGGHHHHHHHHGGGGHHHGGGGGGGGGHHGGGHGHHHHHHHHHHHHHHHHHHHHHGGHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHGHHHHGHHHHGGGGGGHHHHHGGGGGHHHHHHHHHHHHHHHGHGHHHHGGGGHHGGGGGHGGHHHHHHHHHHHHHHHHHHHHHHHHGGGGGHGGGGGGHGGHHHHHHHHHHHHGGGGGGGGGGFFFFFBBBBBBB
@MISEQ03:123:000000000-A40KM:1:1101:16328:1722 1:N:0:TCCACAGGAGT
AGAGGGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCTGTAGTCCGT
+
ABCCBBBCCFFBGGGGGGGGGGGGHHHHHGHGGGGGHHHHGGGGGHHGGGGGHHHHHHHHFGGGHHHGGGGGGGGGHHGGGHGHHHHGHHHHHHHHHHHHGHHHGGHHGHHHHHHHHHGHHHHGHHHHHGHHHHHHHHHHHHHHHHHHHHHHHHHHHGHHHHHHHHHHGHHHHHHGGGGGHHHHHGGGGFHHHGHHHHHHHGHHHFHFGFHHGFFFHHGGGGGHGGHFFFHFHHHHHHHFGHHHHHHHHHGGGGFHGGGECGHGGHHHHHHHHHHHHGGGGGGGGGGFFFFFBBBABBA
@MISEQ03:123:000000000-A40KM:1:1101:17115:1764 1:N:0:TCCACAGGAGT
GTGGGGTGCCAGCAGCCGCGGTAATACGGAGGATCCAAGCGTTATCCGGAATCATTGGGTTTAAAGGGTCCGTAGGCGGTTTAATAAGTCAGTGGTGAAAGCCCATCGCTCAACGGTGGAACGGCCATTGATACTGTTAAACTTGAATTATTAGGAAGTAACTAGAATATGTAGTGTAGCGGTGAAATGCTTAGAGATTACATGGAATACCAATTGCGAAGGCAGGTTACTACTAATGGATTGACGCTGATGGACGAAAGCGTGGGTAGCGAACAGGATTAGATACCCCTGTAGTCCGT
+
CCCCCCBBCFFFGGGGGGGGGGGGHHHHGGGGGHHHHHHGGGGGHHHGGGGGHHHHHHHGGGHHHHHGHHHGHHGGHGGGGHHHHFHHGHHHHHHHHHHHHHHHHHHHHHHGHHHHHHHGGGHHHHHHHHGHGHHHHHHHHHHHHHHHHHHHHGHHHHGHHHHHHHHHHHHHHHHHGGGGHHGGHHHHHHGHHHHHHHHHFHHGHHHHHHHHGGGGHHHHHHHHHHHHHHHHHHHHHHHHGGGGGHHHHHGGGHGHGGGEGGHHHGGGGGHHHHHHHGGGGGGGGGGFFFFFBCCCBBA
@MISEQ03:123:000000000-A40KM:1:1101:14429:1791 1:N:0:TCCACAGGAGT
GTACCGTGCCAGCAGCCGCGGTAATACGTAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGTGCGCAGGCGGTTATGTAAGACAGTTGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGTGACTGCATAGCTAGAGTACGGTAGAGGGGGATGGAATTCCGCGTGTAGCAGTGAAATGCGTAGATATGCGGAGGAACACCGATGGCGAAGGCAATCCCCTGGACCTGTACTGACGCTCATGCACGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCTGTAGTCCGT
+
ABBCCCCCCCFFGGGGGGGGGGGGHHHHHHHGGHGGHHHHGGGGGHHGGGGGHHHHHHHHGGGGHHHGGGGGGGGGGGGFF/BFFDHFF2FFGFGGGGHHHGHHGGFGFFEGHFHFGHGFHFHHGHHGHHHHGHHHHHHHHHHHHHGHFGHHHHGHHHHHHHHHFHGGGGGGGHHHHHHHHHHHHGGGGGHHHHHGGGGFHHHHHGGEFFGHGGGGFHHHHHHHGGGGHHHHFFHHHDHHGGGGCHGHHHHFCHGHGGGGGGHGGHHHHHHHHHHHHGGGGGGGGGGFFFFFBBBBBBA
@MISEQ03:123:000000000-A40KM:1:1101:14969:1845 1:N:0:TCCACAGGAGT
CGCCGGGTGCCAGCCGCCGCGGTAATACGTAGGGGGCTAGCGTTGTCCGGATTTATTGGGCGTAAAGAGCGTGTAGGCGGCCAGGTAGGTCCGGTGTGAAAACTCGAGGCTTAACCTCGAGATGTCATCGGAAACCATCTGGCTAGAGTCCGGAAGAGGAGAGTGGAATTCCTGGTGTAGCGGTGAAATGCGCAGATATCAGGAAGAACACCTATGGCGAAAGCAGCTCTCTGGGACGGTACTGACGCTGAGACGCGAAAGCGTAGGGAGCGAACAGGATTAGATACCCCGGTAGTCCGT
+
AAABBB@ABBFFGGGGGGGGGFGGGGGGHHHGGGGGGFFHGGGGGGHEGGEGGHFHHHHHHGGGGHHHHHGGGGHHGGGFGGFFFHHHGHHHDCADGGHHFF<GHGHGGGHHHGHHHHGGGFGHHHHHHHHHGGGGGGGGGHGHHHGGGGGHHHHHGGGHHHHHHFHHHHGHGHHHGGGGGHHGFHGGGGGGFHGHGGFFHHHHHHFGFFFFHGGGGGHHHHGGGGFEHGFHFFFCGHHHHFGGGFEEHGGGGGGGHGGDGGHGHFDHGGGFHGHHHHGCGGGGGEGGDDBFFDA33?AA
@MISEQ03:123:000000000-A40KM:1:1101:17499:1855 1:N:0:TCCACAGGAGT
AGCCGGTGCCAGCCGCCGCGGTAATACGGAGGATCCAAGCGTTATCCGGAATCATTGGGTTTAAAGGGTCCGTAGGCGGTTTAGTAAGTCAGTGGTGAAAGCCCATCGCTCAACGGTGGAACGGCCATTGATACTGCTAGACTTGAATTATTAGGAAGTAACTAGAATATGTAGTGTAGCGGTGAAATGCTTAGAGATTACATGGAATACCAATTGCGAAGGCAGGTTACTACTAATGGATTGACGCTGATGGACGAAAGCGTGGGTAGCGAACAGGATTAGAAACCCTGGTAGTCCGT
+
CCCCCCCCBFFFGGGGGGGGGGGGHHHHGGGGGHHHHHHGGGGGHHHGGGGGHHHHHHHGGGHHHHGGHGGGHHGGHGGGHHHHHHHHHHHHHHHHHHHHHHHHHHHGGHHGHHGGFBHGHHFHGHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHFHHHHHHHHHHHHHGGGGGHHHHHHHHGGFHHHHHHHGGFHHHHHHHGFGHFGGGHHHHHFHHHHHHGHHHHFHHHHHHGGGGGGHGHHGGGHGHGGGGGGHHHGGGGGHHHFHHHGGGGFEGGGGFFFFFBDAAABA
@MISEQ03:123:000000000-A40KM:1:1101:14009:1889 1:N:0:TCCACAGGAGT
GCGGCGTGCCAGCCGCTGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGTACCCGGGTAGTCCGT
+
ABBBBBBBBBFFGGGGFGGGGGGGHHHHHHHGGGGGHHHHGGGGGHHGGGGGHGHHHHHHGGGGHHHGGGGGGGGGGHGGGHGHHHHHHHHHHHHHHHHHHHHHGGHHHGHHHHHHHHGGHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHGHGHGGGGGHGHHHGGGGGHHHHHHHHGHHHHHHGHGFHHHGGGGGHGGGGFHGGHGHHFHGHHHHFGHHHHFHHHHHGGGGGEHGGGGGGFGGHHGHFHHHHHHGGGGGGGGGGBFFFFBBBBBBB
@MISEQ03:123:000000000-A40KM:1:1101:17869:1908 1:N:0:TCCACAGGAGT
ATTGCGTGCCAGCCGCCGCGGTAATACAGAGGTCCCAAGCGTTGTTCGGATTCACTGGGCGTAAAGGGTGCGTAGGCGGTCGGGTAAGTCTGACGTGAAATCTCCGAGCCTAACTCGGAAACTGCGTCGGATACTATTCGGCTAGAGGAATGGAGGGGAGACTGGAATTCTCGGTGTAGCAGTGAAATGCGTAGATATCGAGAGGAACACCAGTGGCGAAGGCGAGTCTCTGGACATTTCCTGACGCTGAGGCACGAAAGCCAGGGGAGCAAACGGGATTAGAAACCCCGGTAGTCCGT
+
BCCCCCCCCCFFGGGGGGGGGGGGHHHHHHHGHHHHHHHGGGGGGHHGGGGGHHHHHHHHGGGGHHHGHGGGGGGGHGGGGGGGHGHHHHHHGHGGGHGHHHHHHGGGGGHHHHHGGGGGGHHHHGGGGFGFEGHECHGHHHHGHGGHFHHGGHGHGFFHGEFGFHHGGGGGHHHHHHGGHHFGHGGGGGHHHHFHFHGHHHHGHGHHGHHHGGGGHHGGGFGHHHHHGHHHHHHHHHHHGGGGGHHHHHHGGFGHHHHHHGHGGHHHHGHHHGHHHGGGGGFGGGGBBFFFBBBBBBB
@MISEQ03:123:000000000-A40KM:1:1101:14767:1933 1:N:0:TCCACAGGAGT
GCGTGGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGTGCGCAGGCGGTTTTGTAAGACAGTTGTGAAATCCCCGGGCTCAACCTGGGAACTGCCTTTAAAACTGCATCGCTTGAACATGGGAGAGGGGGGTGGAATTCCGCGTGTAGAGGTGAAATTCGTAGATATTCGGAAGAACACCAGTGGCGAAGGCGGCGCACTGGACGACTGTTGACGCTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCGGTAGTCCGT
+
3AAA?BBBFFFFGGGGGGGGGGGGHHHHHHHGGGGGHHHHGGGGGHHGGGGGHHHHHHHHGGGGHHHGFGGGGGGGGGGED??0GFGG1?BC1FFFFEHHHHHHGGBDCGHFHGHCGHG0GFHHHBHGG0G0F0G9<<00:CC/:CF90;/CC;EDGG.@.GDBBGFFF/B//GDHGFBHGDDHFFFFB<EFFFF/FF//FFB2FEF0GFFF/9FFFGEEEEEE@BFE@G9EFFEBF>@1>;9EA1GB0A/?E/GGCEEE/GGFCHHFF0FDGDGDDGGBBGEAE0GDDC>C1DAA>>1
@MISEQ03:123:000000000-A40KM:1:1101:17050:1939 1:N:0:TCCACAGGAGT
GGATGGTGCCAGCCGCCGCGGTAATACGTAGGTGGCAAGCGTTGTCCGGATTTATTGGGTTTAAAGGGTGCGTAGGCGGCCCTGTAAGTCAGTGCTGAAATATCCCGGCTTAACCGGGAGGGTGGCATTGATACTGCGGGGCTTGAGTACAGATGAAGTCGGCGGAATTGACGGTGTAGCGGTGAAATTCATAGATATCGGGAGGAACACCAGTGGCGAAGGCGAGTCTCTGGGCTGATACTGACGCTGAGGCGCGAAAGCCAGGGGAGCAAACGGGATTAGAAACCCCTGTAGTCCGT
+
DDDCCCFFFFFFGGGGGGGGGGGGHHHHHHHGHHGHHHGHGGGGGHHGGEGGHHHHHHHGGGHHHHHGHGGGGGGGHGGGGGHHHHHHHHHHHHGHHHHGHHHFHHGGGGGHHHHGGGGGGGGGGGHHHHHHHHHHHGHGGHHHHHHGHGHGHHHGGGGGGGHHHHHGGGHGHHHGGGGGHHHHHHHHHHHHHHGGGGGGHHHHHFFHHFHHGGGGGHGGGGGHHHHHHGHGHHHHHHHHGGGGGHHHGGGGGGGHHHHHHGHGGHHHHGHHHGHHHGGGGGGEEGGFFFFFBBBBBBB
@MISEQ03:123:000000000-A40KM:1:1101:16782:1940 1:N:0:TCCACAGGAGT
GAGTTGTGCCAGCCGCCGCGGTAATACGTAGGGGGCAAGCGTTGTCCGGATTTATTGGGCGTAAAGAGCGTGTAGGCGGCTAGGTAGGTCCGTTGTGAAAACTCGAGGCTCAACCTCGAGACGTCGATGGAAACCATCTAGCTAGAGTCCGGAAGAGGAGAGTGGAATTCCTGGTGTAGCGGTGAAATGCGCAGATATCAGGAAGAACACCCGTGGCGAAGGCGGCTCTCTGGTACGTGACTGACGCTGAGACGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCTTGTAGTCCGT
+
BCCCCFFCFFFFGGGGGGGGGGGGHGHHHHHGGGGGGGGHGGGGGHHGGGGGHHHHHHHHGGGGHHHHHGGGGHHHHGGGG9GHHHHGHHHGHHGGGHHHFHHGGHGGGHHHHHHHHGGGHGGGGGHHHHHHHHHHHHHHHHHHHGGGGGHGHHHHHHGHHHHHGHGHFGHGHHHGGGGGHHHHHGGGGGHHGHHHHHHHHHHHFFGGGGGGGGGGHHGGGGGHHGHEHHHGHGHGHHHHGGGGGHHHHGGGGGGHGGGGFGHGGHHHHHHHHHHHHGGGGGGGGGGFFFFFBBBBBBB
@MISEQ03:123:000000000-A40KM:1:1101:13534:1945 1:N:0:TCCACAGGAGT
GGGGGGTGCCAGCAGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTCGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGTCGAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCAGTAGTCCGT
+
DDDDDDBCCFFFGGGGGGGGGGGGHHHHHHHGGGGGHHHHGGGGGHHGGGGGGGHHHHHHGGGGHHHGGGGGGGGGGGGGGHGGHHGGGGGGGGGGGGGGGGGGGGHHGGGHHGHHHHGHHHHGFHHGHHFHHGGGGGHHHHHHHHHHHHGHHHHHGGHHHHHHHHHHHHHGHHHGGGGGHHHHHGGGGGHHHHHHHHHHHHHHHFHGGGHHGGGGHHGGGGGHGGHGHHGGHHHHHGHHHHHHHHHHHHGGGGGHGGGGGGHGGHHHHHHHHHHHHGGGGGGEEGGFFFFFBBBBBBB
@MISEQ03:123:000000000-A40KM:1:1101:14333:2023 1:N:0:TCCACAGGAGT
TTGACGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCGGTAGTCCGT
+
BCCCCCCBCCFFGGGGGGGGGGGGHHHHHHHGGGGGHHHHGGGGGHHGGGGGHGHHHHHHGGGGHHHGGGGGGGGGGGGGGHGHHHHHHHHGHHHHHHHHHHHHGGHHGGHHHHHHHHHHHGHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHGGGGGHHGHHGGGGFHHHHHHHHHHHHHHHFHHHHHHGGGGHHGGGGGHGGHHHHHHHHHHHHHHHHHHHHHHHHGGGGGHGGGGCGHGGHHHHHHHHHHHHGGGGGGGGGGCCFFFCBCBBCB
@MISEQ03:123:000000000-A40KM:1:1101:17225:2042 1:N:0:TCCACAGGAGT
TCCCTGTGCCAGCCGCCGCGGTAATACAGAGGTCTCAAGCGTTGTTCGGATTCATTGGGCGTAAAGGGTGCGTAGGTGGCGTGGTAAGTCGGGTGTGAAATTTCGGAGCTTAACTCCGAAACTGCATTCGATACTGCCGTGCTTGAGGACTGGAGAGGAGACTGGAATTTACGGTGTAGCAGTGAAATGCGTAGAGATCGTAAGGAAGACCAGTGGCGAAGGCGGGTCTCTGGACAGTTCCTGACACTGAGGCACGAAGGCCAGGGGAGCAAACGGGATTAGAAACCCTGGTAGTCCGT
+
CCCCCFFFFFFFGGGGGGGGGGGGHHHHHHHGHHHHHHHHGGGGGHHGGGGGHHHHHHHHGGGGHGHGHGGGFGGGGGGHGGGGHHHHHHGGGHGGGHHHGHHGHHGGGHHHHHHHHHHHHHHHHHHHGHHHHGHHGHHHFHHHHHHHHHHHHHHHHHHGHCHHHHGGGHHGHHHHGHGHHHHHHGGGGGHGGHHHHGGHHHHHHHHHHHHHGGGGHHGGGGGHGHGHHHHHHHHHHHHGHHHHHHHHHHHGGHHHHHHHHEHGGHHHHGHHHGHHHGGGGGGGGGGFFFFFBBBBBBB
@MISEQ03:123:000000000-A40KM:1:1101:16728:2094 1:N:0:TCCACAGGAGT
GGGGGGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCGTGTAGTCCGT
+
CCCCCCCCCFFFGGGGGGGGGGGGHHHHHHHGGGGGHHHHGGGGGHHGGGGGHHHHHHHHGGGGHHHGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGHHHFGHHFFGHHHHHHHHHHHGHHHHHHHHHHHHHGHHHHHHGHHHGHHHGGHHGHHGHHHHHGHHHHHGGGGGHHFFHGGGGGHHHHHHHGHHHHHHHGHHHHHHGGGGHHGGGGGHGGHHGHHGHHHHHHHHHHHHHHHHHHGGGGGHGGGGGGHGGGHHHHHHHGHHHGGGGGGGGGGCFFFFCCCCBBA
@MISEQ03:123:000000000-A40KM:1:1101:15092:2113 1:N:0:TCCACAGGAGT
GGGGCGTGCCAGCAGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCAGTAGTCCGT
+
CCDCDDDDDDFFGGGGGGGGGGGGHHHHHHHGGGGGHHHHGGGGGHHGGGGGHGHHHHHHGGGGHHHGGGGGGGGGGHGGGHGHHHHHHHHGHHHHHHHHHHHHGGHGHGHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHGGHHHHFGHHHHHHHHHHHGHGHGHHFHHGGGGFHHHHHHHHHHHHHHHFHGHHHHGFGGHHGGGGGGGGHGHHHHHHHHHGHHHHHHHHHGHHGGGGGHGGGGGGHGGHHHHHHHHHHHHGGGGGGGGGGFFFFFCBCCCCB
@MISEQ03:123:000000000-A40KM:1:1101:15596:2117 1:N:0:TCCACAGGAGT
GTAGGGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCTGGTAGTCCGT
+
AABCCCFABFFFGGGGGGGGGGFGHHHHHHHGGGGGHHHHGGGGGHHGGGGGHHHHHHHHGGGGHHHGGGGGGGGGGGGGGHGHHHHHHHHHHHHHHHHHHHHHGGGGGGHHHHHHHHHHHHHHHGHHHHHHHHHHHHHHHHHHHHHHHHHHHGHHHHHHFHHHHHHHHHGGHHHGGGGGHHGHHGFFFFHHFHHGHGHHHHHHHGHHHHGHFFGGFHGGGGGHGGHHGHHHHFFHHHFHHHHHHHHHHHGGGGGHGGGFCGHGGHHHHHHHHHHHHGGGGGGGGGGFFFFFDBAAAAA
"""

# should be empty
expected_default_un1 = """"""

# should be empty
expected_default_un2 = """"""

expected_report = """202
203
203
203
204
203
203
203
203
203
203
203
202
203
204
203
203
203
203
203
203
203
203
203
203
"""

expected_assembly_alt_param = """@MISEQ03:123:000000000-A40KM:1:1101:14149:1572 1:N:0:TCCACAGGAGT
GGGGGGTGCCAGCCGCCGCGGTAATACGGGGGGGGCAAGCGTTGTTCGGAATTACTGGGCGTAAAGGGCTCGTAGGCGGCCAACTAAGTCAGACGTGAAATCCCTCGGCTTAACCGGGGAACTGCGTCTGATACTGGATGGCTTGAGGTTGGGAGAGGGATGCGGAATTCCAGGTGTAGCGGTGAAATGCGTAGATATCTGGAGGAACACCGGTGGCGAAGGCGGCATCCTGGACCAATTCTGACGCTGAGTGAGCGAAAGCCAGGGGAGCAAACGGGATTAGAAACCCTAGTAGTCCGT
+
CCCCCCCCCFFFGGGGGGGGGGGGHHHHGGGGGFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF-BFFFFFFFFFF9FFFFFFGFGFFGGFFFFGGFFHHHHFFFGGFHHHFFGHHFGHHHHGHGFFGGGGHFHHGHHGHHGFFFEHHGGHFHFGFGHGGGGGHHGHHGGGGGFHGHGGGGFFHHHHGFFGGGGHGGGGGHGGGGGHGHG1GHHHHHHHHHGHGGGGFGHHHHGGHGGGHGEGEHFHGGHHHHEHHHGHHHGGGGFGGGGGFFFFFADBAABB
@MISEQ03:123:000000000-A40KM:1:1101:14170:1596 1:N:0:TCCACAGGAGT
ATGGCGTGCCAGCAGCCGCGGTAATACGGAGGGGGCTAGCGTTGTTCGGAATTACTGGGCGTAAAGCGCACGTAGGCGGCTTTGTAAGTTAGAGGTGAAAGCCCGGGGCTCAACTCCGGAACTGCCTTTAAGACTGCATCGCTAGAATTGTGGAGAGGTGAGTGGAATTCCGAGTGTAGAGGTGAAATTCGTAGATATTCGGAAGAACACCAGTGGCGAAGGCGACTCACTGGACACATATTGACGCTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCGTGTAGTCCGT
+
CCCCCCCCCCFFGGGGGGGGGGGGGHHHGGGGGGGGGHHHGGGGGHHGGGGGHHHHHHHHGGGGHHHGGGGGHHHGHGGGGGHHHHHHHHGHGHHHHHHGHHHHGGGGGHHHHHHHHHHHHGHHHHHHHHHGHHHGGHGGHHHHHHHHHHHHHHHHHHHHHHHHHHGGGGGHHHHHHHHHHHFHHHHFGGHFHHHGGGGGHHHHHGHHFHFHGGGGHHGGGGGGFE5HHHHHHHHHHFHHGGGGGHGHHHGGGGGHGGGGGGHGGHHHGHHHHHHHHGGGGGGGGGGBFFFFBBBBBBB
@MISEQ03:123:000000000-A40KM:1:1101:14740:1607 1:N:0:TCCACAGGAGT
AGTGTGTGCCAGCAGCCGCGGTAATACGTAGGGTGCGAGCGTTAATCGGAATTACTGGGCGTAAAGCGTGCGCAGGCGGTTCGTTGTGTCTGCTGTGAAATCCCCGGGCTCAACCTGGGAATGGCAGTGGAAACTGGCGAGCTTGAGTGTGGCAGAGGGGGGTGGAATTCCGCGTGTAGCAGTGAAATGCGTAGAGATGCGGAGGAACACCGATGGCGAAGGCAACCCCCTGGGATAATATTGACGCTCATGCACGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCAGTAGTCCGT
+
AABCCFFFFFFFGGGGGGGGGGGGHHHHHHHEGGFG2EEGGGGGGHHGGGGGHGHHHHHHGGGGHHHGGGGGGGGGGGGFGGGGHFGFGBGGFHFGFFHHGHHHGGGGFCHHHHHFGGGGGGGHGHGGGGGHH/EDHFGHGHHHHHGFGHGHHHGGGEGGGHHHHHFFFFF/DHHHHHHFHHGFHGGGGGHHHHHGGGGEEHHHHEGGFDGHGGGGHHHGHHEEEFGGGHHFGF5FFFHFGGGGFHHHHHHFEHGHGGGGFGHGGHHCHCHFHGFHHGFGBGGFFGGFFFFFDBBBBAA
@MISEQ03:123:000000000-A40KM:1:1101:14875:1613 1:N:0:TCCACAGGAGT
GGTGGGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCTGTAGTCCGT
+
CCCCCCCCCFFFGGGGGGGGGGGGGHHHHHHGGGGGHHHHGGGGGHHGGGGGHHHHHHHHGGGGHHHGGGGGGGGGHHGGGHGHHHHHHHHHHHHHHHHHGHHHGGHGGGHHHHHHHHHHHHHHHHHHHHHHHHHGHHHHHHHHHHHHHHHHHHHHHHHHHHHHHGHHHHHHHHHGGGGGHGHHHGGGGGHHHHHHHHHHHHHHHGHHHGFHGGFGHHGGGGGHGGHHHHHHHHHHHHHHHHHHHHHHHHGGGGGHGGGGGGHGGHHHHHHHHHHHHGGGGGGGGGGBFFFFBBBBBBB
@MISEQ03:123:000000000-A40KM:1:1101:14902:1627 1:N:0:TCCACAGGAGT
AGGGGGTGCCAGCCGCCGCGGTAAGACGAAGGTGGCTAGCGTTATTCGGATTTACTGGGCATAAAGCGCGCGTAGGCGGCTTGGCAAGTCGGAGGTGAAAGCCTGCAGCTCAACTGCGGTGTGCCTCCGAAACTGCCGAGCTTGAGTGCGGGAGGGGTGATTGGAACTCCAGGTGTAGCGGTGAAATGCGTAGATATCTGGAGGAACACCAGAGGCGAAGGCGAATCACTGGACCGCAACTGACGCTGAGGCGCGAAAGCTAGGGGAGCAAACAGGATTAGAAACCCCGGTAGTCCGT
+
ABBBBBBBBFFCFGGGGGGGGGGGHHHGGGGGHHGHHHHHGGGGHHHHHGEGHHHHHHHHHHHHHHHGGGGGGGGGHGGGGGGGHFHGHHGGGGGGHHHHHHFHHHHHHHHHHHHCGGGGGGGHHHHGGGDGDGGGGHHHGHGHGGGFFGGGGGHGHHHHFHHHHHFFFGHHHHGGGGGHHHGHGGGGGHHHHHHHHFHHHHHHFHHHHFHGGGGHHGGFFFHHHEHHHHGGGFFFGHHGGGGGHFHGGGFEGFHHHHGGGHGGHHGHHHHHHHHHGGGGGEEAGGBBFFFDDBBBBB
@MISEQ03:123:000000000-A40KM:1:1101:15883:1644 1:N:0:TCCACAGGAGT
GGGATGTGCCAGCAGCCGCGGTAATACAGAGGTCCCGAGTGTTGTTCGGATTCACTGGGCGTAAAGGGTGCGTAGGTGGTGGGGTAAGTCTCGTGTGAAAGCTCGCAGCTTAACTGCGAAAAGTCACGGGATACTATTCCGCTGGAGGCTTGGAGGGGGGACTGGAATACTTGGTGTAGCAGTGAAATGCGTAGATATCAAGTGGAACACCAGTGGCGAAGGCGAGTCCCTGGACAAGACCTGACACTGAGGCACGAAAGCCAGGGGAGCAAACAGGATTAGAAACCCTTGTAGTCCGT
+
CDDDDDFFFFFFGGGGGGGGGGGGHHHHHHHGHGGH2EFGHHHHHHHGGGGGHHHHHHHHGGGGHHHGHGGGGGGGHHGGGGGGGGHHHHHHHGHHHHHHHHHGGGHGGHHHHHHDGHHGHHHHHHHHHGGGHHHHHHHHGGGHHHHGGHHGGGHGGHHGHFGHHHHHGGHHHHGHGHHHHHHHHGGGGGHHHHHHHHHHHHHGHFFGFGHHGGGGHHGGGGGHGFDGHGFHHHHHHHHHHHHHHHHHHHHFEHGHHHHGGGHGGHHHHHHHGHHHHGGGGGGGGGGFFFFFBBBBBBB
@MISEQ03:123:000000000-A40KM:1:1101:17346:1668 1:N:0:TCCACAGGAGT
GGCGGGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAACGTTGCTCGGAATCACTGGGCGTAAAGGGCGTGTAGGCGGGAAGGAAAGTCGGGCGTGAAATCCCTGGGCTCAACCCAGGAACTGCACTCGAAACTTCCTTTCTTGAGTGCCGGAGAGGAAAGCGGAATTCCTGGTGTAGAGGTGAAATTCGTAGATATCAGGAGGAACACCAGTGGCGAAGGCGGCTTTCTGGACGGTAACTGACGCTGAGACGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCGGGTAGTCCGT
+
BCCCCDDBCFFFGGGGGGGGGGGGHHHHHGGGGAFEHFHHGHHHCFHGGEGGHHHGHGHHGGGGHHHGEEEGEFHFFGGGBBFF?GGF?FDGFGGCGDGHFHHHHGHGGHGGHHHHBGE;CFGGFCGFFFGFG?CFFGFGGHFFHHDCGFFFA<FGFFFFFDFFGFFGFGGHGFFGD1FFHHHGFFGFGGHGGEGEHHFFHFHHFEGHHHGHFGFFFHGGGFEHFFFHGF?BBGGFDHFHGFEA?HHHHGGGGGGHGGGGGGHEEHHHHHFHHGFD4GGFFGGGGGGACFFFDDAA@BB
@MISEQ03:123:000000000-A40KM:1:1101:14457:1707 1:N:0:TCCACAGGAGT
TGGGTGTGCCAGCCGCCGCGGTAATACGAGGGGTGCAAGCGTTGCTCGGAATTATTGGGCGTAAAGGGTAGGTAGGTGGTCTCATTTGTCTGGGGTGAAAGCCTCGAGCTTAACTCGAGAAGTGCCCCAGAAACGGTGAGACTAGAGTTCTGGGGAGGGTCGTGGAATTCCCGGTGTAGCGGTGAAATGCGTAGAGATCGGGAGGAACACCAGAGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCTGTAGTCCGT
+
AABCBCCCFFFFGGGGGGGGGGGGHHHHGGGGGGGGHHHHGGGGGHHGGFGGHHHHHHHHGGGGHHHGHHGGHHHGHHGGHHFBFHHHHHHHHHGGGHHHHHHHHGGGGGHHHHHHGHGGGHHHHHHHHHHHHHHHGHHHHHHHHHHHHHHHHHHHGGGGGHGHHHHGGGGGHHHGGGGGHHHHHGGGGGHHHHHGGGGGHHHHHGHHHHHHGGGGGHGGGGGHGGHHHHHHHHHHHFHHHHHHHHHHHHGGGGGHGGGGGGHGGHHHHHHHHHHHHGGGGGGGGGGFFFFFBBBCCCB
@MISEQ03:123:000000000-A40KM:1:1101:15133:1720 1:N:0:TCCACAGGAGT
TATAGGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCTTGTAGTCCGT
+
DDDDDFFFFFFFGGGGGGGGGGGGHHHHHHHGGGGGHHHHGGGGGHHGGGGGHHHHHHHHGGGGHHHGGGGGGGGGHHGGGHGHHHHHHHHHHHHHHHHHHHHHGGHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHGHHHHGHHHHGGGGGGHHHHHGGGGGHHHHHHHHHHHHHHHGHGHHHHGGGGHHGGGGGHGGHHHHHHHHHHHHHHHHHHHHHHHHGGGGGHGGGGGGHGGHHHHHHHHHHHHGGGGGGGGGGFFFFFBBBBBBB
@MISEQ03:123:000000000-A40KM:1:1101:16328:1722 1:N:0:TCCACAGGAGT
AGAGGGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCTGTAGTCCGT
+
ABCCBBBCCFFBGGGGGGGGGGGGHHHHHGHGGGGGHHHHGGGGGHHGGGGGHHHHHHHHFGGGHHHGGGGGGGGGHHGGGHGHHHHGHHHHHHHHHHHHGHHHGGHHGHHHHHHHHHGHHHHGHHHHHGHHHHHHHHHHHHHHHHHHHHHHHHHHHGHHHHHHHHHHGHHHHHHGGGGGHHHHHGGGGFHHHGHHHHHHHGHHHFHFGFHHGFFFHHGGGGGHGGHFFFHFHHHHHHHFGHHHHHHHHHGGGGFHGGGECGHGGHHHHHHHHHHHHGGGGGGGGGGFFFFFBBBABBA
@MISEQ03:123:000000000-A40KM:1:1101:17115:1764 1:N:0:TCCACAGGAGT
GTGGGGTGCCAGCAGCCGCGGTAATACGGAGGATCCAAGCGTTATCCGGAATCATTGGGTTTAAAGGGTCCGTAGGCGGTTTAATAAGTCAGTGGTGAAAGCCCATCGCTCAACGGTGGAACGGCCATTGATACTGTTAAACTTGAATTATTAGGAAGTAACTAGAATATGTAGTGTAGCGGTGAAATGCTTAGAGATTACATGGAATACCAATTGCGAAGGCAGGTTACTACTAATGGATTGACGCTGATGGACGAAAGCGTGGGTAGCGAACAGGATTAGATACCCCTGTAGTCCGT
+
CCCCCCBBCFFFGGGGGGGGGGGGHHHHGGGGGHHHHHHGGGGGHHHGGGGGHHHHHHHGGGHHHHHGHHHGHHGGHGGGGHHHHFHHGHHHHHHHHHHHHHHHHHHHHHHGHHHHHHHGGGHHHHHHHHGHGHHHHHHHHHHHHHHHHHHHHGHHHHGHHHHHHHHHHHHHHHHHGGGGHHGGHHHHHHGHHHHHHHHHFHHGHHHHHHHHGGGGHHHHHHHHHHHHHHHHHHHHHHHHGGGGGHHHHHGGGHGHGGGEGGHHHGGGGGHHHHHHHGGGGGGGGGGFFFFFBCCCBBA
@MISEQ03:123:000000000-A40KM:1:1101:14429:1791 1:N:0:TCCACAGGAGT
GTACCGTGCCAGCAGCCGCGGTAATACGTAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGTGCGCAGGCGGTTATGTAAGACAGTTGTGAAATCCCCGGGCTCAACCTGGGAACTGCATCTGTGACTGCATAGCTAGAGTACGGTAGAGGGGGATGGAATTCCGCGTGTAGCAGTGAAATGCGTAGATATGCGGAGGAACACCGATGGCGAAGGCAATCCCCTGGACCTGTACTGACGCTCATGCACGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCTGTAGTCCGT
+
ABBCCCCCCCFFGGGGGGGGGGGGHHHHHHHGGHGGHHHHGGGGGHHGGGGGHHHHHHHHGGGGHHHGGGGGGGGGGGGFF/BFFDHFF2FFGFGGGGHHHGHHGGFGFFEGHFHFGHGFHFHHGHHGHHHHGHHHHHHHHHHHHHGHFGHHHHGHHHHHHHHHFHGGGGGGGHHHHHHHHHHHHGGGGGHHHHHGGGGFHHHHHGGEFFGHGGGGFHHHHHHHGGGGHHHHFFHHHDHHGGGGCHGHHHHFCHGHGGGGGGHGGHHHHHHHHHHHHGGGGGGGGGGFFFFFBBBBBBA
@MISEQ03:123:000000000-A40KM:1:1101:14969:1845 1:N:0:TCCACAGGAGT
CGCCGGGTGCCAGCCGCCGCGGTAATACGTAGGGGGCTAGCGTTGTCCGGATTTATTGGGCGTAAAGAGCGTGTAGGCGGCCAGGTAGGTCCGGTGTGAAAACTCGAGGCTTAACCTCGAGATGTCATCGGAAACCATCTGGCTAGAGTCCGGAAGAGGAGAGTGGAATTCCTGGTGTAGCGGTGAAATGCGCAGATATCAGGAAGAACACCTATGGCGAAAGCAGCTCTCTGGGACGGTACTGACGCTGAGACGCGAAAGCGTAGGGAGCGAACAGGATTAGATACCCCGGTAGTCCGT
+
AAABBB@ABBFFGGGGGGGGGFGGGGGGHHHGGGGGGFFHGGGGGGHEGGEGGHFHHHHHHGGGGHHHHHGGGGHHGGGFGGFFFHHHGHHHDCADGGHHFF<GHGHGGGHHHGHHHHGGGFGHHHHHHHHHGGGGGGGGGHGHHHGGGGGHHHHHGGGHHHHHHFHHHHGHGHHHGGGGGHHGFHGGGGGGFHGHGGFFHHHHHHFGFFFFHGGGGGHHHHGGGGFEHGFHFFFCGHHHHFGGGFEEHGGGGGGGHGGDGGHGHFDHGGGFHGHHHHGCGGGGGEGGDDBFFDA33?AA
@MISEQ03:123:000000000-A40KM:1:1101:17499:1855 1:N:0:TCCACAGGAGT
AGCCGGTGCCAGCCGCCGCGGTAATACGGAGGATCCAAGCGTTATCCGGAATCATTGGGTTTAAAGGGTCCGTAGGCGGTTTAGTAAGTCAGTGGTGAAAGCCCATCGCTCAACGGTGGAACGGCCATTGATACTGCTAGACTTGAATTATTAGGAAGTAACTAGAATATGTAGTGTAGCGGTGAAATGCTTAGAGATTACATGGAATACCAATTGCGAAGGCAGGTTACTACTAATGGATTGACGCTGATGGACGAAAGCGTGGGTAGCGAACAGGATTAGAAACCCTGGTAGTCCGT
+
CCCCCCCCBFFFGGGGGGGGGGGGHHHHGGGGGHHHHHHGGGGGHHHGGGGGHHHHHHHGGGHHHHGGHGGGHHGGHGGGHHHHHHHHHHHHHHHHHHHHHHHHHHHGGHHGHHGGFBHGHHFHGHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHFHHHHHHHHHHHHHGGGGGHHHHHHHHGGFHHHHHHHGGFHHHHHHHGFGHFGGGHHHHHFHHHHHHGHHHHFHHHHHHGGGGGGHGHHGGGHGHGGGGGGHHHGGGGGHHHFHHHGGGGFEGGGGFFFFFBDAAABA
@MISEQ03:123:000000000-A40KM:1:1101:14009:1889 1:N:0:TCCACAGGAGT
GCGGCGTGCCAGCCGCTGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGTACCCGGGTAGTCCGT
+
ABBBBBBBBBFFGGGGFGGGGGGGHHHHHHHGGGGGHHHHGGGGGHHGGGGGHGHHHHHHGGGGHHHGGGGGGGGGGHGGGHGHHHHHHHHHHHHHHHHHHHHHGGHHHGHHHHHHHHGGHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHGHGHGGGGGHGHHHGGGGGHHHHHHHHGHHHHHHGHGFHHHGGGGGHGGGGFHGGHGHHFHGHHHHFGHHHHFHHHHHGGGGGEHGGGGGGFGGHHGHFHHHHHHGGGGGGGGGGBFFFFBBBBBBB
@MISEQ03:123:000000000-A40KM:1:1101:17869:1908 1:N:0:TCCACAGGAGT
ATTGCGTGCCAGCCGCCGCGGTAATACAGAGGTCCCAAGCGTTGTTCGGATTCACTGGGCGTAAAGGGTGCGTAGGCGGTCGGGTAAGTCTGACGTGAAATCTCCGAGCCTAACTCGGAAACTGCGTCGGATACTATTCGGCTAGAGGAATGGAGGGGAGACTGGAATTCTCGGTGTAGCAGTGAAATGCGTAGATATCGAGAGGAACACCAGTGGCGAAGGCGAGTCTCTGGACATTTCCTGACGCTGAGGCACGAAAGCCAGGGGAGCAAACGGGATTAGAAACCCCGGTAGTCCGT
+
BCCCCCCCCCFFGGGGGGGGGGGGHHHHHHHGHHHHHHHGGGGGGHHGGGGGHHHHHHHHGGGGHHHGHGGGGGGGHGGGGGGGHGHHHHHHGHGGGHGHHHHHHGGGGGHHHHHGGGGGGHHHHGGGGFGFEGHECHGHHHHGHGGHFHHGGHGHGFFHGEFGFHHGGGGGHHHHHHGGHHFGHGGGGGHHHHFHFHGHHHHGHGHHGHHHGGGGHHGGGFGHHHHHGHHHHHHHHHHHGGGGGHHHHHHGGFGHHHHHHGHGGHHHHGHHHGHHHGGGGGFGGGGBBFFFBBBBBBB
@MISEQ03:123:000000000-A40KM:1:1101:17050:1939 1:N:0:TCCACAGGAGT
GGATGGTGCCAGCCGCCGCGGTAATACGTAGGTGGCAAGCGTTGTCCGGATTTATTGGGTTTAAAGGGTGCGTAGGCGGCCCTGTAAGTCAGTGCTGAAATATCCCGGCTTAACCGGGAGGGTGGCATTGATACTGCGGGGCTTGAGTACAGATGAAGTCGGCGGAATTGACGGTGTAGCGGTGAAATTCATAGATATCGGGAGGAACACCAGTGGCGAAGGCGAGTCTCTGGGCTGATACTGACGCTGAGGCGCGAAAGCCAGGGGAGCAAACGGGATTAGAAACCCCTGTAGTCCGT
+
DDDCCCFFFFFFGGGGGGGGGGGGHHHHHHHGHHGHHHGHGGGGGHHGGEGGHHHHHHHGGGHHHHHGHGGGGGGGHGGGGGHHHHHHHHHHHHGHHHHGHHHFHHGGGGGHHHHGGGGGGGGGGGHHHHHHHHHHHGHGGHHHHHHGHGHGHHHGGGGGGGHHHHHGGGHGHHHGGGGGHHHHHHHHHHHHHHGGGGGGHHHHHFFHHFHHGGGGGHGGGGGHHHHHHGHGHHHHHHHHGGGGGHHHGGGGGGGHHHHHHGHGGHHHHGHHHGHHHGGGGGGEEGGFFFFFBBBBBBB
@MISEQ03:123:000000000-A40KM:1:1101:16782:1940 1:N:0:TCCACAGGAGT
GAGTTGTGCCAGCCGCCGCGGTAATACGTAGGGGGCAAGCGTTGTCCGGATTTATTGGGCGTAAAGAGCGTGTAGGCGGCTAGGTAGGTCCGTTGTGAAAACTCGAGGCTCAACCTCGAGACGTCGATGGAAACCATCTAGCTAGAGTCCGGAAGAGGAGAGTGGAATTCCTGGTGTAGCGGTGAAATGCGCAGATATCAGGAAGAACACCCGTGGCGAAGGCGGCTCTCTGGTACGTGACTGACGCTGAGACGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCTTGTAGTCCGT
+
BCCCCFFCFFFFGGGGGGGGGGGGHGHHHHHGGGGGGGGHGGGGGHHGGGGGHHHHHHHHGGGGHHHHHGGGGHHHHGGGG9GHHHHGHHHGHHGGGHHHFHHGGHGGGHHHHHHHHGGGHGGGGGHHHHHHHHHHHHHHHHHHHGGGGGHGHHHHHHGHHHHHGHGHFGHGHHHGGGGGHHHHHGGGGGHHGHHHHHHHHHHHFFGGGGGGGGGGHHGGGGGHHGHEHHHGHGHGHHHHGGGGGHHHHGGGGGGHGGGGFGHGGHHHHHHHHHHHHGGGGGGGGGGFFFFFBBBBBBB
@MISEQ03:123:000000000-A40KM:1:1101:13534:1945 1:N:0:TCCACAGGAGT
GGGGGGTGCCAGCAGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTCGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGTCGAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCCAGTAGTCCGT
+
DDDDDDBCCFFFGGGGGGGGGGGGHHHHHHHGGGGGHHHHGGGGGHHGGGGGGGHHHHHHGGGGHHHGGGGGGGGGGGGGGHGGHHGGGGGGGGGGGGGGGGGGGGHHGGGHHGHHHHGHHHHGFHHGHHFHHGGGGGHHHHHHHHHHHHGHHHHHGGHHHHHHHHHHHHHGHHHGGGGGHHHHHGGGGGHHHHHHHHHHHHHHHFHGGGHHGGGGHHGGGGGHGGHGHHGGHHHHHGHHHHHHHHHHHHGGGGGHGGGGGGHGGHHHHHHHHHHHHGGGGGGEEGGFFFFFBBBBBBB
@MISEQ03:123:000000000-A40KM:1:1101:14333:2023 1:N:0:TCCACAGGAGT
TTGACGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCGGTAGTCCGT
+
BCCCCCCBCCFFGGGGGGGGGGGGHHHHHHHGGGGGHHHHGGGGGHHGGGGGHGHHHHHHGGGGHHHGGGGGGGGGGGGGGHGHHHHHHHHGHHHHHHHHHHHHGGHHGGHHHHHHHHHHHGHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHGGGGGHHGHHGGGGFHHHHHHHHHHHHHHHFHHHHHHGGGGHHGGGGGHGGHHHHHHHHHHHHHHHHHHHHHHHHGGGGGHGGGGCGHGGHHHHHHHHHHHHGGGGGGGGGGCCFFFCBCBBCB
@MISEQ03:123:000000000-A40KM:1:1101:17225:2042 1:N:0:TCCACAGGAGT
TCCCTGTGCCAGCCGCCGCGGTAATACAGAGGTCTCAAGCGTTGTTCGGATTCATTGGGCGTAAAGGGTGCGTAGGTGGCGTGGTAAGTCGGGTGTGAAATTTCGGAGCTTAACTCCGAAACTGCATTCGATACTGCCGTGCTTGAGGACTGGAGAGGAGACTGGAATTTACGGTGTAGCAGTGAAATGCGTAGAGATCGTAAGGAAGACCAGTGGCGAAGGCGGGTCTCTGGACAGTTCCTGACACTGAGGCACGAAGGCCAGGGGAGCAAACGGGATTAGAAACCCTGGTAGTCCGT
+
CCCCCFFFFFFFGGGGGGGGGGGGHHHHHHHGHHHHHHHHGGGGGHHGGGGGHHHHHHHHGGGGHGHGHGGGFGGGGGGHGGGGHHHHHHGGGHGGGHHHGHHGHHGGGHHHHHHHHHHHHHHHHHHHGHHHHGHHGHHHFHHHHHHHHHHHHHHHHHHGHCHHHHGGGHHGHHHHGHGHHHHHHGGGGGHGGHHHHGGHHHHHHHHHHHHHGGGGHHGGGGGHGHGHHHHHHHHHHHHGHHHHHHHHHHHGGHHHHHHHHEHGGHHHHGHHHGHHHGGGGGGGGGGFFFFFBBBBBBB
@MISEQ03:123:000000000-A40KM:1:1101:16728:2094 1:N:0:TCCACAGGAGT
GGGGGGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGAAACCCGTGTAGTCCGT
+
CCCCCCCCCFFFGGGGGGGGGGGGHHHHHHHGGGGGHHHHGGGGGHHGGGGGHHHHHHHHGGGGHHHGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGGHHHFGHHFFGHHHHHHHHHHHGHHHHHHHHHHHHHGHHHHHHGHHHGHHHGGHHGHHGHHHHHGHHHHHGGGGGHHFFHGGGGGHHHHHHHGHHHHHHHGHHHHHHGGGGHHGGGGGHGGHHGHHGHHHHHHHHHHHHHHHHHHGGGGGHGGGGGGHGGGHHHHHHHGHHHGGGGGGGGGGCFFFFCCCCBBA
@MISEQ03:123:000000000-A40KM:1:1101:15092:2113 1:N:0:TCCACAGGAGT
GGGGCGTGCCAGCAGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAAGCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCCAGTAGTCCGT
+
CCDCDDDDDDFFGGGGGGGGGGGGHHHHHHHGGGGGHHHHGGGGGHHGGGGGHGHHHHHHGGGGHHHGGGGGGGGGGHGGGHGHHHHHHHHGHHHHHHHHHHHHGGHGHGHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHHGGHHHHFGHHHHHHHHHHHGHGHGHHFHHGGGGFHHHHHHHHHHHHHHHFHGHHHHGFGGHHGGGGGGGGHGHHHHHHHHHGHHHHHHHHHGHHGGGGGHGGGGGGHGGHHHHHHHHHHHHGGGGGGGGGGFFFFFCBCCCCB
@MISEQ03:123:000000000-A40KM:1:1101:15596:2117 1:N:0:TCCACAGGAGT
GTAGGGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGCGCGTAGGTGGTTTGTTAAGTTGGATGTGAAATCCCCGGGCTCAACCTGGGAACTGCATTCAAAACTGACAAGCTAGAGTATGGTAGAGGGTGGTGGAATTTCCTGTGTAGCGGTGAAATGCGTAGATATAGGAAGGAACACCAGTGGCGAAGGCGACCACCTGGACTGATACTGACACTGAGGTGCGAAAGCGTGGGGAGCAAACAGGATTAGATACCCTGGTAGTCCGT
+
AABCCCFABFFFGGGGGGGGGGFGHHHHHHHGGGGGHHHHGGGGGHHGGGGGHHHHHHHHGGGGHHHGGGGGGGGGGGGGGHGHHHHHHHHHHHHHHHHHHHHHGGGGGGHHHHHHHHHHHHHHHGHHHHHHHHHHHHHHHHHHHHHHHHHHHGHHHHHHFHHHHHHHHHGGHHHGGGGGHHGHHGFFFFHHFHHGHGHHHHHHHGHHHHGHFFGGFHGGGGGHGGHHGHHHHFFHHHFHHHHHHHHHHHGGGGGHGGGFCGHGGHHHHHHHHHHHHGGGGGGGGGGFFFFFDBAAAAA
"""

expected_default_un1_alt_param = """@MISEQ03:123:000000000-A40KM:1:1101:14767:1933 1:N:0:TCCACAGGAGT
GCGTGGTGCCAGCCGCCGCGGTAATACAGAGGGTGCAAGCGTTAATCGGAATTACTGGGCGTAAAGCGTGCGCAGGCGGTTTTGTAAGACAGTTGTGAAATCCCCGGGCTCAACCTGGGAACTGCCTTTAAAACTGCATCGCTTGAACATGGGAGAGGGGGGTGGAATTCCGAGTGTAGCGGTGAAATTCGTAGAGATTCGGAAGAACACCAGTGGCGAAGGCGGCCCCCTGGACTAATATTGACGCTGAG
+
3AAA?BBBFFFFGGGGGGGGGGGGHHHHHHHGGGGGHHHHGGGGGHHGGGGGHHHHHHHHGGGGHHHGFGGGGGGGGGGED??0GFGG1?BC1@FFDEHHHHHHGG@DCGHFHGHCGHG0GFHHHBHGG0G0=0G0<<00::C.:CF00;//.;EDGG.@-;@BBEFFF/B/.;;AFFBB9BABFFFFB9EFFFF/BF//99A.;.F.;FFE.9FFFBBBBDE9@B-9-;9EF.9/B/;/9;9B9.A####
"""

expected_default_un2_alt_param = """@MISEQ03:123:000000000-A40KM:1:1101:14767:1933 2:N:0:TCCACAGGAGT
ACGGACTACCGGGGTTTCTAATCCTGTTTGCTCCCCACGCTTTCGCACCTCAGCGTCAACAGTCGTCCAGTGCGCCGCCTTCGCCACTGGTGTTCTTCCGAATATCTACGAATTTCACCTCTACACGCGGAATTCCACCCCCCTCTCCCACGTTCAAGCTATGCAGTCTTACAGGCAGTTCCCAGGTTGAGCCCGGGGATTTCACAACTCTCTTACGAAACCGCCTGCGCACGCTTCACCCCCAGTAATTC
+
1>>AAD1C>CDDG0EAEGBBGGDDGDGDF0FFHHCFGG/EEECGG/E?/A0BG1AE//>1@>FBEFE>/G@EF@?EEEEEEGFEA//F1<G0FEF2BFF//?F/BDD2?<//?FFHDDGH0DGHDG//<-><CG0=DG.<--;C.:CC/;.9.;9/0C0/0;99B0F/00C00999C.:0F0C90-;FF/99B-9@---;BBFEFF#############################################
"""


if __name__ == '__main__':
    main()
